/***************************************************************************

Gottlieb driver : dedicated to Warren Davis, Jeff Lee & David Thiel

driver by Fabrice Frances

****************************************************************************

Reactor: earlier version of the board, with a different memory map

Main processor (8088 minimum mode)
0000-1fff RAM (NOT battery backed unlike all the others)
2000-2fff sprites
3000-37ff video ram
4000-4fff char generator ram
6000-67ff palette ram (palette of 16 colors)
7000-77ff i/o ports
8000-ffff ROM

memory mapped ports:

read:
7000    Dip switch
7001    Inputs 10-17
7002    trackball H
7003    trackball V
7004    Inputs 40-47

write:
7000    watchdog reset
7001    trackball clear
7002    Outputs 20-27
7003    Flipflop outputs:
		b0: F/B priority
		b1: horiz. flip
		b2: vert. flip
		b3: Output 33
		b4: coin counter
		b5: left lamp (1P/3Lives start)
		b6: middle lamps (2P/3Lives start, 1P/7Lives start)
		b7: right lamp (2P/7Lives start)
7004    Outputs 40-47

interrupts:
INTR not connected
NMI connected to vertical blank



Gottlieb games memory map

Main processor (8088 minimum mode)
0000-0fff RAM (battery backed)
1000-1fff RAM or ROM (selected with jumpers on the board)
2000-2fff RAM or ROM (selected with jumpers on the board)
3000-37ff sprites. The manual says there are 63 sprites (NOT 64),
          but the Q*Bert object priority test leaves sprite #63 dangling, so
		  they are probably only 62.
3800-3fff video RAM
4000-4fff char generator RAM (can be replaced by a ROM twice as large,
          selection made with jumpers on the board. If it's ROM, the CPU
		  cannot fully access it, I think it could read half the data if it
		  wanted to but none of the games do that)
5000-57ff palette ram (palette of 16 colors)
5800-5fff i/o ports
6000-ffff ROM (not necessarily fully populated)

memory mapped ports:

read:
5800    Dip switch
5801    Inputs 10-17
5802    trackball H (optional)
5803    trackball V (optional)
5804    Inputs 40-47

write:
5800    watchdog reset
5801    trackball clear (optional)
5802    Outputs 20-27
5803    Flipflop outputs:
		b0: F/B priority
		b1: horiz. flip (sprite bank in Us vs. Them)
		b2: vert. flip (maybe genlock control in the laser disc games)
		b3: Output 33
		b4: coin counter (sprite bank in Q*Bert Qubes)
		b5: Q*Bert: kicker; Q*Bert Qubes: coin counter
		b5/b6: 3 Stooges: joystick input multiplexer
		b7: ?
5804    Outputs 40-47

interrupts:
INTR not connected
NMI connected to vertical blank



Sound processor (6502) memory map (earlier revision, used by games up to Krull):
0000-0fff RIOT (6532)
1000-1fff amplitude DAC
2000-2fff SC01 voice chip
3000-3fff voice clock DAC
4000-4fff expansion socket
5000-5fff expansion socket
6000-6fff expansion socket or ROM (selected with jumpers on the board)
7000-7fff ROM
(repeated in 8000-ffff, A15 only used in expansion socket)

Use of I/Os on the RIOT:
both ports A and B are programmed as inputs, A is connected to the main
motherboard, and B has SW1 (test) connected on bit 6.

interrupts:
INTR is connected to the RIOT, so an INTR can be generated by a variety
of sources, e.g active edge detection on PA7, or timer countdown.
It seems that all gottlieb games program the interrupt conditions so that
a positive active edge on PA7 triggers an interrupt, so the
main board ensures a command is correctly received by sending nul (0)
commands between two commands. Also, the timer interrupt is enabled but
doesn't seem to serve any purpose...(?)


In the later revision of the sound board, used from M.A.C.H. 3 onwards, there
are two 6502, two 8910, a DAC and a GI SP-0250 speech chip.


Video timings:
XTAL = 20 MHz
Horizontal video frequency: HSYNC = XTAL/4/318 = 15.72327 kHz
Video frequency: VSYNC = HSYNC/256 = 61.41903 Hz
VBlank duration: 1/VSYNC * (16/256) = 1017.6 us

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"



int gottlieb_vh_start(void);
void gottlieb_vh_stop(void);
void gottlieb_characterram_w(int offset, int data);
void gottlieb_video_outputs(int offset,int data);
void usvsthem_video_outputs(int offset,int data);
extern unsigned char *gottlieb_characterram;
void gottlieb_paletteram_w(int offset,int data);
void gottlieb_vh_screenrefresh(struct osd_bitmap *bitmap,int full_refresh);

void gottlieb_sh_w(int offset, int data);

extern unsigned char *riot_ram;
int riot_ram_r(int offset);
int gottlieb_riot_r(int offset);
void riot_ram_w(int offset, int data);
void gottlieb_riot_w(int offset, int data);
void gottlieb_speech_w(int offset, int data);
void gottlieb_speech_clock_DAC_w(int offset, int data);
void gottlieb_sound_init(void);
int stooges_sound_input_r(int offset);
void stooges_8910_latch_w(int offset,int data);
void stooges_sound_control_w(int offset,int data);
void gottlieb_nmi_rate_w(int offset,int data);
void gottlieb_cause_dac_nmi_w(int offset, int data);



static int track[2];

int gottlieb_track_0_r(int offset)
{
	return input_port_2_r(offset) - track[0];
}

int gottlieb_track_1_r(int offset)
{
	return input_port_3_r(offset) - track[1];
}

void gottlieb_track_reset_w(int offset,int data)
{
	/* reset the trackball counters */
	track[0] = input_port_2_r(offset);
	track[1] = input_port_3_r(offset);
}

static int joympx;

int stooges_IN4_r(int offset)
{
	int joy;


	switch (joympx)
	{
		case 0:
		default:
			joy = ((readinputport(4) >> 0) & 0x0f);	/* joystick 1 */
			break;
		case 1:
			joy = ((readinputport(5) >> 0) & 0x0f);	/* joystick 2 */
			break;
		case 2:
			joy = ((readinputport(5) >> 4) & 0x0f);	/* joystick 3 */
			break;
	}

	return joy | (readinputport(4) & 0xf0);
}

void reactor_output(int offset,int data)
{
	osd_led_w(0,(data & 0x20) >> 5);
	osd_led_w(1,(data & 0x40) >> 6);
	osd_led_w(2,(data & 0x80) >> 7);
	gottlieb_video_outputs(offset,data);
}

void stooges_output(int offset,int data)
{
	joympx = (data >> 5) & 0x03;
	gottlieb_video_outputs(offset,data);
}


static int current_frame = 0x00001;
static int laserdisc_playing;
static int lasermpx;

int gottlieb_laserdisc_status_r(int offset)
{
	switch (offset)
	{
		case 0:
			return (current_frame >> 0) & 0xff;
			break;
		case 1:
			return (current_frame >> 8) & 0xff;
			break;
		case 2:
			if (lasermpx == 1)
				/* bits 0-2 frame number MSN */
				/* bit 3 audio buffer ready */
				/* bit 4 ready to send new laserdisc command? */
				/* bit 5 disc ready */
				/* bit 6 break in audio trasmission */
				/* bit 7 missing audio clock */
				return ((current_frame >> 16) & 0x07) | 0x10 | (rand() & 0x28);
			else	/* read audio buffer */
				return rand();
			break;
	}

	return 0;
}

void gottlieb_laserdisc_mpx_w(int offset,int data)
{
	lasermpx = data & 1;
}

void gottlieb_laserdisc_command_w(int offset,int data)
{
	static int loop;
	int cmd;
	static int lastcmd;


	/* commands are written in three steps, the first two the command is */
	/* written (maybe one to load the latch, the other to start the send), */
	/* the third 0 (maybe to clear the latch) */
	if (data == 0) return;
	if (loop++ & 1) return;

	if ((data & 0xe0) != 0x20)
	{
if (errorlog) fprintf(errorlog,"error: laserdisc command %02x\n",data);
		return;
	}

	cmd =	((data & 0x10) >> 4) |
			((data & 0x08) >> 2) |
			((data & 0x04) >> 0) |
			((data & 0x02) << 2) |
			((data & 0x01) << 4);

if (errorlog) fprintf(errorlog,"laserdisc command %02x -> %02x\n",data,cmd);
	if (lastcmd == 0x0b && (cmd & 0x10))	/* seek frame # */
	{
		current_frame = (current_frame << 4) | (cmd & 0x0f);
	}
	else
	{
		if (cmd == 0x04)	/* step forward */
		{
			laserdisc_playing = 0;
			current_frame++;
		}
		if (cmd == 0x05) laserdisc_playing = 1;	/* play */
		if (cmd == 0x0f) laserdisc_playing = 0;	/* stop */
		if (cmd == 0x0b) laserdisc_playing = 0;	/* seek frame */
		lastcmd = cmd;
	}
}

int gottlieb_interrupt(void)
{
	if (laserdisc_playing) current_frame++;

	return nmi_interrupt();
}


static unsigned char *nvram;
static int nvram_size;

static void nvram_handler(void *file,int read_or_write)
{
	if (read_or_write)
		osd_fwrite(file,nvram,nvram_size);
	else
	{
		if (file)
			osd_fread(file,nvram,nvram_size);
		else
			memset(nvram,0xff,nvram_size);
	}
}



static struct MemoryReadAddress reactor_readmem[] =
{
	{ 0x0000, 0x1fff, MRA_RAM },
	{ 0x3000, 0x33ff, MRA_RAM },
	{ 0x4000, 0x4fff, MRA_RAM },
	{ 0x7000, 0x7000, input_port_0_r },	/* DSW */
	{ 0x7001, 0x7001, input_port_1_r },	/* buttons */
	{ 0x7002, 0x7002, gottlieb_track_0_r },	/* trackball H */
	{ 0x7003, 0x7003, gottlieb_track_1_r },	/* trackball V */
	{ 0x7004, 0x7004, input_port_4_r },	/* joystick */
	{ 0x8000, 0xffff, MRA_ROM },
	{ -1 }  /* end of table */
};

static struct MemoryWriteAddress reactor_writemem[] =
{
	{ 0x0000, 0x1fff, MWA_RAM },
	{ 0x2000, 0x20ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x3000, 0x33ff, videoram_w, &videoram, &videoram_size },
	{ 0x3400, 0x37ff, videoram_w },	/* mirror address, some games write to it */
	{ 0x4000, 0x4fff, gottlieb_characterram_w, &gottlieb_characterram },
	{ 0x6000, 0x601f, gottlieb_paletteram_w, &paletteram },
	{ 0x7000, 0x7000, watchdog_reset_w },
	{ 0x7001, 0x7001, gottlieb_track_reset_w },
	{ 0x7002, 0x7002, gottlieb_sh_w }, /* sound/speech command */
	{ 0x7003, 0x7003, reactor_output },       /* OUT1 */
	{ 0x8000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};

static struct MemoryReadAddress gottlieb_readmem[] =
{
	{ 0x0000, 0x0fff, MRA_RAM },
	{ 0x1000, 0x1fff, MRA_RAM },	/* or ROM */
	{ 0x2000, 0x2fff, MRA_RAM },	/* or ROM */
	{ 0x3800, 0x3bff, MRA_RAM },
	{ 0x4000, 0x4fff, MRA_RAM },
	{ 0x5800, 0x5800, input_port_0_r },	/* DSW */
	{ 0x5801, 0x5801, input_port_1_r },	/* buttons */
	{ 0x5802, 0x5802, gottlieb_track_0_r },	/* trackball H */
	{ 0x5803, 0x5803, gottlieb_track_1_r },	/* trackball V */
	{ 0x5804, 0x5804, input_port_4_r },	/* joystick */
	{ 0x5805, 0x5807, gottlieb_laserdisc_status_r },
	{ 0x6000, 0xffff, MRA_ROM },
	{ -1 }  /* end of table */
};

static struct MemoryWriteAddress gottlieb_writemem[] =
{
	{ 0x0000, 0x0fff, MWA_RAM, &nvram, &nvram_size },
	{ 0x1000, 0x1fff, MWA_RAM },	/* ROM in Krull */
	{ 0x2000, 0x2fff, MWA_RAM },	/* ROM in Krull and 3 Stooges */
	{ 0x3000, 0x30ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x3800, 0x3bff, videoram_w, &videoram, &videoram_size },
	{ 0x3c00, 0x3fff, videoram_w },	/* mirror address, some games write to it */
	{ 0x4000, 0x4fff, gottlieb_characterram_w, &gottlieb_characterram },
	{ 0x5000, 0x501f, gottlieb_paletteram_w, &paletteram },
	{ 0x5800, 0x5800, watchdog_reset_w },
	{ 0x5801, 0x5801, gottlieb_track_reset_w },
	{ 0x5802, 0x5802, gottlieb_sh_w }, /* sound/speech command */
	{ 0x5803, 0x5803, gottlieb_video_outputs },       /* OUT1 */
	{ 0x6000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};


/* same as above, different video_outputs plus laser disc control outputs */
static struct MemoryWriteAddress usvsthem_writemem[] =
{
	{ 0x0000, 0x0fff, MWA_RAM, &nvram, &nvram_size },
	{ 0x1000, 0x1fff, MWA_RAM },	/* ROM in Krull */
	{ 0x2000, 0x2fff, MWA_RAM },	/* ROM in Krull and 3 Stooges */
	{ 0x3000, 0x30ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x3800, 0x3bff, videoram_w, &videoram, &videoram_size },
	{ 0x3c00, 0x3fff, videoram_w },	/* mirror address, some games write to it */
	{ 0x4000, 0x4fff, gottlieb_characterram_w, &gottlieb_characterram },
	{ 0x5000, 0x501f, gottlieb_paletteram_w, &paletteram },
	{ 0x5800, 0x5800, watchdog_reset_w },
	{ 0x5801, 0x5801, gottlieb_track_reset_w },
	{ 0x5802, 0x5802, gottlieb_sh_w }, /* sound/speech command */
	{ 0x5803, 0x5803, usvsthem_video_outputs },       /* OUT1 */
	{ 0x5805, 0x5805, gottlieb_laserdisc_command_w },	/* command for the player */
	{ 0x5806, 0x5806, gottlieb_laserdisc_mpx_w },
	{ 0x6000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};

/* same as above, different IN4 */
static struct MemoryReadAddress stooges_readmem[] =
{
	{ 0x0000, 0x0fff, MRA_RAM },
	{ 0x1000, 0x1fff, MRA_RAM },
	{ 0x2000, 0x2fff, MRA_ROM },
	{ 0x3800, 0x3bff, MRA_RAM },
	{ 0x4000, 0x4fff, MRA_RAM },
	{ 0x5800, 0x5800, input_port_0_r },	/* DSW */
	{ 0x5801, 0x5801, input_port_1_r },	/* buttons */
	{ 0x5802, 0x5802, gottlieb_track_0_r },	/* trackball H */
	{ 0x5803, 0x5803, gottlieb_track_1_r },	/* trackball V */
	{ 0x5804, 0x5804, stooges_IN4_r },	/* joystick */
	{ 0x6000, 0xffff, MRA_ROM },
	{ -1 }  /* end of table */
};

/* same as above, different video_outputs */
static struct MemoryWriteAddress stooges_writemem[] =
{
	{ 0x0000, 0x0fff, MWA_RAM, &nvram, &nvram_size },
	{ 0x1000, 0x1fff, MWA_RAM },
	{ 0x2000, 0x2fff, MWA_ROM },
	{ 0x3000, 0x30ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x3800, 0x3bff, videoram_w, &videoram, &videoram_size },
	{ 0x3c00, 0x3fff, videoram_w },	/* mirror address, some games write to it */
	{ 0x4000, 0x4fff, gottlieb_characterram_w, &gottlieb_characterram },
	{ 0x5000, 0x501f, gottlieb_paletteram_w, &paletteram },
	{ 0x5800, 0x5800, watchdog_reset_w },
	{ 0x5801, 0x5801, gottlieb_track_reset_w },
	{ 0x5802, 0x5802, gottlieb_sh_w }, /* sound/speech command */
	{ 0x5803, 0x5803, stooges_output },       /* OUT1 */
	{ 0x6000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};



struct MemoryReadAddress gottlieb_sound_readmem[] =
{
	{ 0x0000, 0x01ff, riot_ram_r },
	{ 0x0200, 0x03ff, gottlieb_riot_r },
	{ 0x6000, 0x7fff, MRA_ROM },
			 /* A15 not decoded except in expansion socket */
	{ 0x8000, 0x81ff, riot_ram_r },
	{ 0x8200, 0x83ff, gottlieb_riot_r },
	{ 0xe000, 0xffff, MRA_ROM },
	{ -1 }  /* end of table */
};

struct MemoryWriteAddress gottlieb_sound_writemem[] =
{
	{ 0x0000, 0x01ff, riot_ram_w, &riot_ram },
	{ 0x0200, 0x03ff, gottlieb_riot_w },
	{ 0x1000, 0x1000, DAC_data_w },
	{ 0x2000, 0x2000, gottlieb_speech_w },
	{ 0x3000, 0x3000, gottlieb_speech_clock_DAC_w },
	{ 0x6000, 0x7fff, MWA_ROM },
			 /* A15 not decoded except in expansion socket */
	{ 0x8000, 0x81ff, riot_ram_w },
	{ 0x8200, 0x83ff, gottlieb_riot_w },
	{ 0x9000, 0x9000, DAC_data_w },
	{ 0xa000, 0xa000, gottlieb_speech_w },
	{ 0xb000, 0xb000, gottlieb_speech_clock_DAC_w },
	{ 0xe000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};


static struct MemoryReadAddress stooges_sound_readmem[] =
{
	{ 0x0000, 0x03ff, MRA_RAM },
	{ 0x8000, 0x8000, soundlatch_r },
	{ 0xe000, 0xffff, MRA_ROM },
	{ -1 }  /* end of table */
};

struct MemoryWriteAddress stooges_sound_writemem[] =
{
	{ 0x0000, 0x03ff, MWA_RAM },
	{ 0x4000, 0x4001, DAC_data_w },
	{ 0xe000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};


static struct MemoryReadAddress stooges_sound2_readmem[] =
{
	{ 0x0000, 0x03ff, MRA_RAM },
	{ 0x6000, 0x6000, stooges_sound_input_r },	/* various signals */
	{ 0xa800, 0xa800, soundlatch_r },
	{ 0xc000, 0xffff, MRA_ROM },
	{ -1 }  /* end of table */
};

struct MemoryWriteAddress stooges_sound2_writemem[] =
{
	{ 0x0000, 0x03ff, MWA_RAM },
	{ 0x2000, 0x2000, MWA_NOP },	/* speech chip. The game sends strings */
									/* of 15 bytes (clocked by 4000). The chip also */
									/* checks a DATA REQUEST bit in 6000. */
	{ 0x4000, 0x4000, stooges_sound_control_w },
	{ 0x8000, 0x8000, stooges_8910_latch_w },
	{ 0xa000, 0xa000, gottlieb_nmi_rate_w },	/* the timer generates NMIs */
	{ 0xb000, 0xb000, gottlieb_cause_dac_nmi_w },
	{ 0xc000, 0xffff, MWA_ROM },
	{ -1 }  /* end of table */
};



INPUT_PORTS_START( reactor )
	PORT_START	/* DSW */
	PORT_DIPNAME( 0x01, 0x01, "Sound with Logos" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Bounce Chambers Points" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPSETTING(    0x02, "15" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Sound with Instructions" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0xc0, 0xc0, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x40, "12000" )
	PORT_DIPSETTING(    0xc0, "15000" )
	PORT_DIPSETTING(    0x80, "20000" )

	PORT_START	/* IN1 */
	PORT_BITX(0x01, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT ( 0xfc, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H */
	PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_X, 15, 20, 0, 0 )

	PORT_START	/* trackball V */
	PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_Y, 15, 20, 0, 0 )

	PORT_START	/* IN4 */
	PORT_BIT ( 0x01, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT ( 0x02, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT ( 0x04, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT ( 0x08, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT ( 0x10, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT ( 0x20, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT ( 0xc0, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( mplanets )
	PORT_START	/* DSW */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x02, "12000" )
	PORT_DIPNAME( 0x08, 0x00, "Allow Round Select" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x14, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x14, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x20, "5" )
	PORT_DIPNAME( 0xc0, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x40, "Easy" )
	PORT_DIPSETTING(    0x00, "Medium" )
	PORT_DIPSETTING(    0x80, "Hard" )
	PORT_DIPSETTING(    0xc0, "Hardest" )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x3c, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x40, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V (dial) */
	PORT_ANALOGX( 0xff, 0x00, IPT_DIAL, 5, 10, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_BUTTON2 )
INPUT_PORTS_END

INPUT_PORTS_START( qbert )
	PORT_START      /* DSW */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Kicker" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_BITX( 0x08, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Auto Round Advance", IP_KEY_NONE, IP_JOY_NONE )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, "SW5" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, "SW7" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "SW8" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
	PORT_BIT( 0xe0, IP_ACTIVE_LOW, IPT_UNUSED )
/* 0x40 must be connected to the IP16 line */

	PORT_START      /* buttons */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_SERVICE( 0x40, IP_ACTIVE_LOW )
	PORT_BITX(0x80, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* joystick */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
INPUT_PORTS_END

INPUT_PORTS_START( qbertqub )
	PORT_START      /* DSW */
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x35, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x24, "A 2/1 B 2/1" )
	PORT_DIPSETTING(    0x14, "A 1/1 B 4/1" )
	PORT_DIPSETTING(    0x30, "A 1/1 B 3/1" )
	PORT_DIPSETTING(    0x10, "A 1/1 B 2/1" )
	PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
	PORT_DIPSETTING(    0x11, "A 2/3 B 2/1" )
	PORT_DIPSETTING(    0x15, "A 1/2 B 3/1" )
	PORT_DIPSETTING(    0x20, "A 1/2 B 2/1" )
	PORT_DIPSETTING(    0x21, "A 1/2 B 1/1" )
	PORT_DIPSETTING(    0x31, "A 1/2 B 1/5" )
	PORT_DIPSETTING(    0x04, "A 1/3 B 2/1" )
	PORT_DIPSETTING(    0x05, "A 1/3 B 1/1" )
	PORT_DIPSETTING(    0x35, DEF_STR( Free_Play ) )
/* 0x25 DEF_STR( 2C_1C )
   0x01 DEF_STR( 1C_1C )
   0x34 DEF_STR( Free_Play ) */
	PORT_DIPNAME( 0x02, 0x00, "1st Bonus Life" )
	PORT_DIPSETTING(    0x00, "10000" )
	PORT_DIPSETTING(    0x02, "15000" )
	PORT_DIPNAME( 0x40, 0x00, "Additional Bonus Life" )
	PORT_DIPSETTING(    0x00, "20000" )
	PORT_DIPSETTING(    0x40, "25000" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x80, "Hard" )

	PORT_START      /* buttons */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_SERVICE( 0x40, IP_ACTIVE_LOW )
	PORT_BITX(0x80, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* joystick */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( krull )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x02, "Hard" )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPNAME( 0x14, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x14, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x20, 0x00, "Hexagon" )
	PORT_DIPSETTING(    0x00, "Roving" )
	PORT_DIPSETTING(    0x20, "Stationary" )
	PORT_DIPNAME( 0xc0, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x40, "30000 30000" )
	PORT_DIPSETTING(    0x00, "30000 50000" )
	PORT_DIPSETTING(    0x80, "40000 50000" )
	PORT_DIPSETTING(    0xc0, "50000 75000" )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START2 )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP     | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT  | IPF_8WAY )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN   | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT   | IPF_8WAY )
INPUT_PORTS_END

INPUT_PORTS_START( mach3 )
	PORT_START      /* DSW0 */
	/* TODO: values are different for 5 lives */
	PORT_DIPNAME( 0x09, 0x08, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x02, "Hard" )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPNAME( 0x24, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "10000 10000" )
	PORT_DIPSETTING(    0x04, "10000 20000" )
	PORT_DIPSETTING(    0x20, "10000 40000" )
	PORT_DIPSETTING(    0x24, "20000 60000" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_BITX(    0x80, 0x80, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Infinite Lives", IP_KEY_NONE, IP_JOY_NONE )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START2 )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( usvsthem )
	PORT_START      /* DSW0 */
	/* TODO: values are different for 5 lives */
	PORT_DIPNAME( 0x09, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Free_Play ) )
/*	PORT_DIPSETTING(    0x09, DEF_STR( Free_Play ) ) */
	PORT_DIPNAME( 0x02, 0x00, "Unknown" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x04, "Hard" )
	PORT_DIPNAME( 0x10, 0x00, "Unknown" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, "Unknown" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, "Unknown" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "Unknown" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_START2 )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( 3stooges )
	PORT_START	/* DSW */
	PORT_DIPNAME (0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING (   0x01, DEF_STR( Off ) )
	PORT_DIPSETTING (   0x00, DEF_STR( On ) )
	PORT_DIPNAME (0x02, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING (   0x00, "Normal" )
	PORT_DIPSETTING (   0x02, "Hard" )
	PORT_DIPNAME (0x08, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING (   0x00, "3" )
	PORT_DIPSETTING (   0x08, "5" )
	PORT_DIPNAME (0x14, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING (   0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING (   0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING (   0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING (   0x14, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x20, 0x00, "Unknown" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME (0x40, 0x00, "1st Bonus Life at" )
	PORT_DIPSETTING (   0x40, "10000" )
	PORT_DIPSETTING (   0x00, "20000" )
	PORT_DIPNAME (0x80, 0x00, "And Bonus Life Every" )
	PORT_DIPSETTING (   0x80, "10000" )
	PORT_DIPSETTING (   0x00, "20000" )

	PORT_START	/* IN1 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* joystick 2 (Moe) */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	/* the bottom four bits of the previous port are multiplexed among */
	/* three joysticks - the following port contains settings for the other two */
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
INPUT_PORTS_END

INPUT_PORTS_START( curvebal )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x08, 0x00, "2 Players Game" )
	PORT_DIPSETTING(    0x08, "1 Credit" )
	PORT_DIPSETTING(    0x00, "2 Credits" )
	PORT_DIPNAME( 0x11, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x10, "Medium" )
	PORT_DIPSETTING(    0x01, "Hard" )
	PORT_DIPSETTING(    0x11, "Hardest" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, "Coins" )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x20, "French" )
	/* TODO: coinage is different when French is selected */
/* PORT_DIPNAME( 0xc2, 0x00, "French Coinage" )
PORT_DIPSETTING(    0x42, "A 3/1 B 1/2" )
PORT_DIPSETTING(    0x82, "A 1/5 B 1/2" )
PORT_DIPSETTING(    0x02, "A 2/1 B 2/3" )
PORT_DIPSETTING(    0xc0, "A 2/1 B 2/1" )
PORT_DIPSETTING(    0x80, "A 1/1 B 1/2" )
PORT_DIPSETTING(    0x40, "A 1/1 B 1/3" )
PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
PORT_DIPSETTING(    0xc2, DEF_STR( Free_Play ) ) */
	PORT_DIPNAME( 0xc2, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x42, "A 4/1 B 1/1" )
	PORT_DIPSETTING(    0x82, "A 3/1 B 1/1" )
	PORT_DIPSETTING(    0x02, "A 2/1 B 1/1" )
	PORT_DIPSETTING(    0xc0, "A 2/1 B 2/1" )
	PORT_DIPSETTING(    0x80, "A 2/1 B 1/2" )
	PORT_DIPSETTING(    0x40, "A 2/1 B 1/3" )
	PORT_DIPSETTING(    0x00, "A 1/1 B 1/1" )
	PORT_DIPSETTING(    0xc2, DEF_STR( Free_Play ) )

	PORT_START      /* IN0 */
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_SERVICE, "Select in Service Mode", KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START	/* trackball H not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* trackball V not used */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER1, "Swing", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX(0x04, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2, "Pitch Left", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_PLAYER2, "Pitch Right", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BITX(0x40, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_PLAYER1, "Bunt", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END



/* the games can store char gfx data in either a 4k RAM area (128 chars), or */
/* a 8k ROM area (256 chars). */
static struct GfxLayout charRAMlayout =
{
	8,8,    /* 8*8 characters */
	128,    /* 128 characters */
	4,      /* 4 bits per pixel */
	{ 0, 1, 2, 3 },
	{ 0, 4, 8, 12, 16, 20, 24, 28},
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32 },
	32*8    /* every char takes 32 consecutive bytes */
};

static struct GfxLayout charROMlayout =
{
	8,8,    /* 8*8 characters */
	256,    /* 256 characters */
	4,      /* 4 bits per pixel */
	{ 0, 1, 2, 3 },
	{ 0, 4, 8, 12, 16, 20, 24, 28},
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32 },
	32*8    /* every char takes 32 consecutive bytes */
};

static struct GfxLayout spritelayout =
{
	16,16,  /* 16*16 sprites */
	256,    /* 256 sprites */
	4,      /* 4 bits per pixel */
	{ 0, 256*32*8, 2*256*32*8, 3*256*32*8 },
	{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16,
			8*16, 9*16, 10*16, 11*16, 12*16, 13*16, 14*16, 15*16 },
	32*8    /* every sprite takes 32 consecutive bytes */
};

static struct GfxLayout qbertqub_spritelayout =
{
	16,16,  /* 16*16 sprites */
	512,    /* 512 sprites */
	4,      /* 4 bits per pixel */
	{ 0, 512*32*8, 2*512*32*8, 3*512*32*8 },
	{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16,
			8*16, 9*16, 10*16, 11*16, 12*16, 13*16, 14*16, 15*16 },
	32*8    /* every sprite takes 32 consecutive bytes */
};

static struct GfxDecodeInfo charRAM_gfxdecodeinfo[] =
{
	{ 0,           0x4000, &charRAMlayout, 0, 1 },	/* the game dynamically modifies this */
	{ REGION_GFX2, 0x0000, &spritelayout,  0, 1 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo charROM_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x0000, &charROMlayout, 0, 1 },
	{ REGION_GFX2, 0x0000, &spritelayout,  0, 1 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo qbertqub_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x0000, &charROMlayout,         0, 1 },
	{ REGION_GFX2, 0x0000, &qbertqub_spritelayout, 0, 1 },
	{ -1 } /* end of array */
};



static struct DACinterface dac1_interface =
{
	1,
	{ 50 }
};

static struct DACinterface dac2_interface =
{
	2,
	{ 25, 25 }
};

static const char *reactor_sample_names[] =
{
	"*reactor",
	"fx_53.wav", /* "8 left" */
	"fx_54.wav", /* "16 left" */
	"fx_55.wav", /* "24 left" */
	"fx_56.wav", /* "32 left" */
	"fx_57.wav", /* "40 left" */
	"fx_58.wav", /* "warning, core unstable" */
	"fx_59.wav", /* "bonus" */
	"fx_31.wav", /* "chamber activated" */
	"fx_39a.wav", /* "2000" */
	"fx_39b.wav", /* "5000" */
	"fx_39c.wav", /* "10000" */
	"fx_39d.wav", /* "15000" */
	"fx_39e.wav", /* "20000" */
	"fx_39f.wav", /* "25000" */
	"fx_39g.wav", /* "30000" */
	"fx_39h.wav", /* "35000" */
	"fx_39i.wav", /* "40000" */
	"fx_39j.wav", /* "45000" */
	"fx_39k.wav", /* "50000" */
	"fx_39l.wav", /* "55000" */
     0	/* end of array */
};

static const char *qbert_sample_names[] =
{
	"*qbert",
	"fx_17a.wav", /* random speech, voice clock 255 */
	"fx_17b.wav", /* random speech, voice clock 255 */
	"fx_17c.wav", /* random speech, voice clock 255 */
	"fx_17d.wav", /* random speech, voice clock 255 */
	"fx_17e.wav", /* random speech, voice clock 255 */
	"fx_17f.wav", /* random speech, voice clock 255 */
	"fx_17g.wav", /* random speech, voice clock 255 */
	"fx_17h.wav", /* random speech, voice clock 255 */
	"fx_18a.wav", /* random speech, voice clock 176 */
	"fx_18b.wav", /* random speech, voice clock 176 */
	"fx_18c.wav", /* random speech, voice clock 176 */
	"fx_18d.wav", /* random speech, voice clock 176 */
	"fx_18e.wav", /* random speech, voice clock 176 */
	"fx_18f.wav", /* random speech, voice clock 176 */
	"fx_18g.wav", /* random speech, voice clock 176 */
	"fx_18h.wav", /* random speech, voice clock 176 */
	"fx_19a.wav", /* random speech, voice clock 128 */
	"fx_19b.wav", /* random speech, voice clock 128 */
	"fx_19c.wav", /* random speech, voice clock 128 */
	"fx_19d.wav", /* random speech, voice clock 128 */
	"fx_19e.wav", /* random speech, voice clock 128 */
	"fx_19f.wav", /* random speech, voice clock 128 */
	"fx_19g.wav", /* random speech, voice clock 128 */
	"fx_19h.wav", /* random speech, voice clock 128 */
	"fx_20a.wav", /* random speech, voice clock 96 */
	"fx_20b.wav", /* random speech, voice clock 96 */
	"fx_20c.wav", /* random speech, voice clock 96 */
	"fx_20d.wav", /* random speech, voice clock 96 */
	"fx_20e.wav", /* random speech, voice clock 96 */
	"fx_20f.wav", /* random speech, voice clock 96 */
	"fx_20g.wav", /* random speech, voice clock 96 */
	"fx_20h.wav", /* random speech, voice clock 96 */
	"fx_21a.wav", /* random speech, voice clock 62 */
	"fx_21b.wav", /* random speech, voice clock 62 */
	"fx_21c.wav", /* random speech, voice clock 62 */
	"fx_21d.wav", /* random speech, voice clock 62 */
	"fx_21e.wav", /* random speech, voice clock 62 */
	"fx_21f.wav", /* random speech, voice clock 62 */
	"fx_21g.wav", /* random speech, voice clock 62 */
	"fx_21h.wav", /* random speech, voice clock 62 */
	"fx_22.wav", /* EH2 with decreasing voice clock */
	"fx_23.wav", /* O1 with varying voice clock */
	"fx_28.wav", /* "hello, I'm ready" */
	"fx_36.wav", /* "byebye" */
	"knocker.wav",
	0	/* end of array */
};

static struct Samplesinterface samples_interface =
{
	1,	/* one channel */
	100	/* volume */
};

static struct AY8910interface ay8910_interface =
{
	2,	/* 2 chips */
	2000000,	/* 2 MHz */
	{ 25, 25 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 }
};



/********************************************************************
*
*  Machine Driver macro
*  ====================
*
*  There are two versions of the machine driver,
*  one for revision 1 sound board (1x6502 + DAC + Votrax),
*  one for revision 2 (2x6502 + DAC + GI SP-0250 speech chip + 2x8910).
*  They are identical apart form the sound CPU subsections.
*
********************************************************************/

#define MACHINE_DRIVER_SOUND_1(GAMENAME,READMEM,WRITEMEM,GFX,NVRAM)	\
static struct MachineDriver machine_driver_##GAMENAME =             \
{                                                                   \
	/* basic machine hardware */                                	\
	{		                                                        \
		{	  	                                                    \
			CPU_I86,												\
			5000000,        /* 5 Mhz */								\
			READMEM,WRITEMEM,0,0,									\
			gottlieb_interrupt,1									\
		},		                                                    \
		{		                                                    \
			CPU_M6502 | CPU_AUDIO_CPU ,								\
			3579545/4,	/* the board can be set to /2 as well */	\
			gottlieb_sound_readmem,gottlieb_sound_writemem,0,0,		\
			ignore_interrupt,1	/* IRQs are triggered by the main CPU */		\
								/* NMIs are triggered by the Votrax SC-01 */	\
		}                                                   		\
	},                                                          	\
	61, 1018,	/* frames per second, vblank duration */			\
	1,	/* 1 CPU slice per frame - interleaving is forced when a sound command is written */ \
	0,						                                    	\
																	\
	/* video hardware */                                        	\
	32*8, 32*8, { 0*8, 32*8-1, 0*8, 30*8-1 },						\
	GFX,                                                        	\
	16, 16,		                                                	\
	0,									                           	\
																	\
	VIDEO_TYPE_RASTER|VIDEO_SUPPORTS_DIRTY|VIDEO_MODIFIES_PALETTE,	\
	0,                                                          	\
	gottlieb_vh_start,												\
	gottlieb_vh_stop,												\
	gottlieb_vh_screenrefresh,										\
																	\
	/* sound hardware */                                        	\
	0,0,0,0,                                                    	\
	{                                                           	\
		{                                                   		\
			SOUND_DAC,												\
			&dac1_interface											\
		},															\
		{															\
			SOUND_SAMPLES,	/* for Votrax simulation */				\
			&samples_interface										\
		}                                                   		\
	},                                                           	\
																	\
	NVRAM															\
}

#define MACHINE_DRIVER_SOUND_2(GAMENAME,READMEM,WRITEMEM,GFX,NVRAM)	\
static struct MachineDriver machine_driver_##GAMENAME =				\
{																	\
	/* basic machine hardware */									\
	{																\
		{															\
			CPU_I86,												\
			5000000,        /* 5 Mhz */								\
			READMEM,WRITEMEM,0,0,									\
			gottlieb_interrupt,1									\
		},															\
		{															\
			CPU_M6502 | CPU_AUDIO_CPU ,								\
			1000000,	/* 1 MHz */									\
			stooges_sound_readmem,stooges_sound_writemem,0,0,		\
			ignore_interrupt,1	/* IRQs are triggered by the main CPU */			\
								/* NMIs are triggered by the second sound CPU */	\
		},															\
		{															\
			CPU_M6502 | CPU_AUDIO_CPU ,								\
			1000000,	/* 1 MHz */									\
			stooges_sound2_readmem,stooges_sound2_writemem,0,0,		\
			ignore_interrupt,1	/* IRQs are triggered by the main CPU */			\
								/* NMIs are triggered by a programmable timer */	\
		}															\
	},																\
	61, 1018,	/* frames per second, vblank duration */			\
	1,	/* 1 CPU slice per frame - interleaving is forced when a sound command is written */ \
	0,																\
																	\
	/* video hardware */											\
	32*8, 32*8, { 0*8, 32*8-1, 0*8, 30*8-1 },						\
	GFX,															\
	16, 16,															\
	0,																\
																	\
	VIDEO_TYPE_RASTER|VIDEO_SUPPORTS_DIRTY|VIDEO_MODIFIES_PALETTE,	\
	0,																\
	gottlieb_vh_start,												\
	gottlieb_vh_stop,												\
	gottlieb_vh_screenrefresh,										\
																	\
	/* sound hardware */											\
	0,0,0,0,														\
	{																\
		{															\
			SOUND_DAC,												\
			&dac2_interface											\
		},															\
		{															\
			SOUND_AY8910,											\
			&ay8910_interface										\
		}															\
	},																\
																	\
	NVRAM															\
}

/* games using the revision 1 sound board */
MACHINE_DRIVER_SOUND_1(reactor,reactor_readmem,reactor_writemem,charRAM_gfxdecodeinfo,0);
MACHINE_DRIVER_SOUND_1(gottlieb,gottlieb_readmem,gottlieb_writemem,charROM_gfxdecodeinfo,nvram_handler);
MACHINE_DRIVER_SOUND_1(qbertqub,gottlieb_readmem,gottlieb_writemem,qbertqub_gfxdecodeinfo,nvram_handler);
MACHINE_DRIVER_SOUND_1(krull,gottlieb_readmem,gottlieb_writemem,charRAM_gfxdecodeinfo,nvram_handler);
/* games using the revision 2 sound board */
MACHINE_DRIVER_SOUND_2(mach3,gottlieb_readmem,usvsthem_writemem,charROM_gfxdecodeinfo,nvram_handler);
MACHINE_DRIVER_SOUND_2(usvsthem,gottlieb_readmem,usvsthem_writemem,qbertqub_gfxdecodeinfo,nvram_handler);
MACHINE_DRIVER_SOUND_2(stooges,stooges_readmem,stooges_writemem,charRAM_gfxdecodeinfo,nvram_handler);


/***************************************************************************

  Game driver(s)

***************************************************************************/

ROM_START( reactor )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "rom7",         0x8000, 0x1000, 0xa62d86fd )
	ROM_LOAD( "rom6",         0x9000, 0x1000, 0x6ed841f3 )
	ROM_LOAD( "rom5",         0xa000, 0x1000, 0xd90576a3 )
	ROM_LOAD( "rom4",         0xb000, 0x1000, 0x0155daae )
	ROM_LOAD( "rom3",         0xc000, 0x1000, 0xf8881385 )
	ROM_LOAD( "rom2",         0xd000, 0x1000, 0x3caba35b )
	ROM_LOAD( "rom1",         0xe000, 0x1000, 0x944e1ddf )
	ROM_LOAD( "rom0",         0xf000, 0x1000, 0x55930aed )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "snd1",         0xf000, 0x800, 0xd958a0fd )
	ROM_RELOAD(               0x7000, 0x800) /* A15 is not decoded */
	ROM_LOAD( "snd2",         0xf800, 0x800, 0x5dc86942 )
	ROM_RELOAD(               0x7800, 0x800) /* A15 is not decoded */

	/* no gfx1 (RAM is used) */

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	/* 0000-0fff empty */
	ROM_LOAD( "fg0",          0x1000, 0x1000, 0xd1f20e15 )	/* sprites */
	/* 2000-2fff empty */
	ROM_LOAD( "fg1",          0x3000, 0x1000, 0x18396c57 )
	/* 4000-4fff empty */
	ROM_LOAD( "fg2",          0x5000, 0x1000, 0x5489605a )
	/* 6000-6fff empty */
	ROM_LOAD( "fg3",          0x7000, 0x1000, 0x8416ad53 )
ROM_END

ROM_START( mplanets )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "rom4",         0x6000, 0x2000, 0x5402077f )
	ROM_LOAD( "rom3",         0x8000, 0x2000, 0x5d18d740 )
	ROM_LOAD( "rom2",         0xa000, 0x2000, 0x960c3bb1 )
	ROM_LOAD( "rom1",         0xc000, 0x2000, 0xeb515f10 )
	ROM_LOAD( "rom0",         0xe000, 0x2000, 0x74de78aa )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "snd1",         0xf000, 0x800, 0x453193a1 )
	ROM_RELOAD(               0x7000, 0x800) /* A15 is not decoded */
	ROM_LOAD( "snd2",         0xf800, 0x800, 0xf5ffc98f )
	ROM_RELOAD(               0x7800, 0x800) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "bg0",          0x0000, 0x1000, 0x709aa24c )	/* chars */
	ROM_LOAD( "bg1",          0x1000, 0x1000, 0x4921e345 )

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "fg3",          0x0000, 0x2000, 0xc990b39f )	/* sprites */
	ROM_LOAD( "fg2",          0x2000, 0x2000, 0x735e2522 )
	ROM_LOAD( "fg1",          0x4000, 0x2000, 0x6456cc1c )
	ROM_LOAD( "fg0",          0x6000, 0x2000, 0xa920e325 )
ROM_END

ROM_START( qbert )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "qb-rom2.bin",  0xa000, 0x2000, 0xfe434526 )
	ROM_LOAD( "qb-rom1.bin",  0xc000, 0x2000, 0x55635447 )
	ROM_LOAD( "qb-rom0.bin",  0xe000, 0x2000, 0x8e318641 )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, 0x15787c07 )
	ROM_RELOAD(               0x7000, 0x800) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, 0x58437508 )
	ROM_RELOAD(               0x7800, 0x800) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, 0x7a9ba824 )	/* chars */
	ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, 0x22e5b891 )

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qb-fg3.bin",   0x0000, 0x2000, 0xdd436d3a )	/* sprites */
	ROM_LOAD( "qb-fg2.bin",   0x2000, 0x2000, 0xf69b9483 )
	ROM_LOAD( "qb-fg1.bin",   0x4000, 0x2000, 0x224e8356 )
	ROM_LOAD( "qb-fg0.bin",   0x6000, 0x2000, 0x2f695b85 )
ROM_END

ROM_START( qbertjp )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "qbj-rom2.bin", 0xa000, 0x2000, 0x67bb1cb2 )
	ROM_LOAD( "qbj-rom1.bin", 0xc000, 0x2000, 0xc61216e7 )
	ROM_LOAD( "qbj-rom0.bin", 0xe000, 0x2000, 0x69679d5c )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, 0x15787c07 )
	ROM_RELOAD(               0x7000, 0x800) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, 0x58437508 )
	ROM_RELOAD(               0x7800, 0x800) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, 0x7a9ba824 )	/* chars */
	ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, 0x22e5b891 )

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qb-fg3.bin",   0x0000, 0x2000, 0xdd436d3a )	/* sprites */
	ROM_LOAD( "qb-fg2.bin",   0x2000, 0x2000, 0xf69b9483 )
	ROM_LOAD( "qb-fg1.bin",   0x4000, 0x2000, 0x224e8356 )
	ROM_LOAD( "qb-fg0.bin",   0x6000, 0x2000, 0x2f695b85 )
ROM_END

ROM_START( sqbert )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "qb-rom2.bin",  0xa000, 0x2000, 0x1e3d4038 )
	ROM_LOAD( "qb-rom1.bin",  0xc000, 0x2000, 0xeaf3076c )
	ROM_LOAD( "qb-rom0.bin",  0xe000, 0x2000, 0x61260a7e )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, 0x15787c07 )
	ROM_RELOAD(               0x7000, 0x800) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, 0x58437508 )
	ROM_RELOAD(               0x7800, 0x800) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qb-bg0.bin",   0x0000, 0x1000, 0xc3118eef )	/* chars */
	ROM_LOAD( "qb-bg1.bin",   0x1000, 0x1000, 0x4f6d8075 )

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qb-fg3.bin",   0x0000, 0x2000, 0xee595eda )	/* sprites */
	ROM_LOAD( "qb-fg2.bin",   0x2000, 0x2000, 0x59884c78 )
	ROM_LOAD( "qb-fg1.bin",   0x4000, 0x2000, 0x2a60e3ad )
	ROM_LOAD( "qb-fg0.bin",   0x6000, 0x2000, 0xb11ad9d8 )
ROM_END

ROM_START( qbertqub )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "qq-rom3.bin",  0x8000, 0x2000, 0xc4dbdcd7 )
	ROM_LOAD( "qq-rom2.bin",  0xa000, 0x2000, 0x21a6c6cc )
	ROM_LOAD( "qq-rom1.bin",  0xc000, 0x2000, 0x63e6c43d )
	ROM_LOAD( "qq-rom0.bin",  0xe000, 0x2000, 0x8ddbe438 )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "qb-snd1.bin",  0xf000, 0x800, 0x15787c07 )
	ROM_RELOAD(               0x7000, 0x800) /* A15 is not decoded */
	ROM_LOAD( "qb-snd2.bin",  0xf800, 0x800, 0x58437508 )
	ROM_RELOAD(               0x7800, 0x800) /* A15 is not decoded */

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qq-bg0.bin",   0x0000, 0x1000, 0x050badde )	/* chars */
	ROM_LOAD( "qq-bg1.bin",   0x1000, 0x1000, 0x8875902f )

	ROM_REGION( 0x10000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "qq-fg3.bin",   0x0000, 0x4000, 0x91a949cc )	/* sprites */
	ROM_LOAD( "qq-fg2.bin",   0x4000, 0x4000, 0x782d9431 )
	ROM_LOAD( "qq-fg1.bin",   0x8000, 0x4000, 0x71c3ac4c )
	ROM_LOAD( "qq-fg0.bin",   0xc000, 0x4000, 0x6192853f )
ROM_END

ROM_START( krull )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "ram2.bin",     0x1000, 0x1000, 0x302feadf )
	ROM_LOAD( "ram4.bin",     0x2000, 0x1000, 0x79355a60 )
	ROM_LOAD( "rom4.bin",     0x6000, 0x2000, 0x2b696394 )
	ROM_LOAD( "rom3.bin",     0x8000, 0x2000, 0x14b0ee42 )
	ROM_LOAD( "rom2.bin",     0xa000, 0x2000, 0xb5fad94a )
	ROM_LOAD( "rom1.bin",     0xc000, 0x2000, 0x1ad956a3 )
	ROM_LOAD( "rom0.bin",     0xe000, 0x2000, 0xa466afae )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "snd1.bin",     0xe000, 0x1000, 0xdd2b30b4 )
	ROM_RELOAD(               0x6000, 0x1000) /* A15 is not decoded */
	ROM_LOAD( "snd2.bin",     0xf000, 0x1000, 0x8cab901b )
	ROM_RELOAD(               0x7000, 0x1000) /* A15 is not decoded */

	/* no gfx1 (RAM is used) */

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "fg3.bin",      0x0000, 0x2000, 0x82d77a45 )	/* sprites */
	ROM_LOAD( "fg2.bin",      0x2000, 0x2000, 0x25a24317 )
	ROM_LOAD( "fg1.bin",      0x4000, 0x2000, 0x7e3ad7b0 )
	ROM_LOAD( "fg0.bin",      0x6000, 0x2000, 0x7402dc19 )
ROM_END

ROM_START( mach3 )
	ROM_REGION( 0x10000, REGION_CPU1 )	/* 64k for code */
	ROM_LOAD( "m3rom4.bin",   0x6000, 0x2000, 0x8bfd5a44 )
	ROM_LOAD( "m3rom3.bin",   0x8000, 0x2000, 0xb1b045af )
	ROM_LOAD( "m3rom2.bin",   0xa000, 0x2000, 0xfbdfb03d )
	ROM_LOAD( "m3rom1.bin",   0xc000, 0x2000, 0x3b0ba80b )
	ROM_LOAD( "m3rom0.bin",   0xe000, 0x2000, 0x70c12bf4 )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "m3drom1.bin",  0xf000, 0x1000, 0xa6e29212 )

	ROM_REGION( 0x10000, REGION_CPU3 )	/* 64k for second sound cpu */
	ROM_LOAD( "m3yrom1.bin",  0xf000, 0x1000, 0xeddf8872 )

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "mach3bg0.bin", 0x0000, 0x1000, 0xea2f5257 )
	ROM_LOAD( "mach3bg1.bin", 0x1000, 0x1000, 0xf543e4ce )

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "mach3fg3.bin", 0x0000, 0x2000, 0x472128b4 )
	ROM_LOAD( "mach3fg2.bin", 0x2000, 0x2000, 0x2a59e99e )
	ROM_LOAD( "mach3fg1.bin", 0x4000, 0x2000, 0x9b88767b )
	ROM_LOAD( "mach3fg0.bin", 0x6000, 0x2000, 0x0bae12a5 )
ROM_END

ROM_START( usvsthem )
	ROM_REGION( 0x10000, REGION_CPU1 )	/* 64k for code */
	ROM_LOAD( "usvs.rm4",     0x6000, 0x2000, 0x0d7a4072 )
	ROM_LOAD( "usvs.rm3",     0x8000, 0x2000, 0x6f32a671 )
	ROM_LOAD( "usvs.rm2",     0xa000, 0x2000, 0x36770716 )
	ROM_LOAD( "usvs.rm1",     0xc000, 0x2000, 0x697bc989 )
	ROM_LOAD( "usvs.rm0",     0xe000, 0x2000, 0x30cf6bd9 )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "usvsdrom.1",   0xe000, 0x2000, 0xc0b5cab0 )

	ROM_REGION( 0x10000, REGION_CPU3 )	/* 64k for second sound cpu */
	ROM_LOAD( "usvsyrom.1",   0xe000, 0x2000, 0xc3d245ca )

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "usvs.bg0",     0x0000, 0x1000, 0x8a0de09f )
	ROM_LOAD( "usvs.bg1",     0x1000, 0x1000, 0x6fb64d3c )

	ROM_REGION( 0x10000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "usvs.fg3",     0x0000, 0x4000, 0x98703015 )
	ROM_LOAD( "usvs.fg2",     0x4000, 0x4000, 0xd3990707 )
	ROM_LOAD( "usvs.fg1",     0x8000, 0x4000, 0xa2057430 )
	ROM_LOAD( "usvs.fg0",     0xc000, 0x4000, 0x7734e53f )
ROM_END

ROM_START( 3stooges )
	ROM_REGION( 0x10000, REGION_CPU1 )     /* 64k for code */
	ROM_LOAD( "gv113ram.4",   0x2000, 0x1000, 0x533bff2a )
	ROM_LOAD( "gv113rom.4",   0x6000, 0x2000, 0x8b6e52b8 )
	ROM_LOAD( "gv113rom.3",   0x8000, 0x2000, 0xb816d8c4 )
	ROM_LOAD( "gv113rom.2",   0xa000, 0x2000, 0xb45b2a79 )
	ROM_LOAD( "gv113rom.1",   0xc000, 0x2000, 0x34ab051e )
	ROM_LOAD( "gv113rom.0",   0xe000, 0x2000, 0xab124329 )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "drom1",        0xe000, 0x2000, 0x87a9fa10 )

	ROM_REGION( 0x10000, REGION_CPU3 )	/* 64k for second sound cpu */
	ROM_LOAD( "yrom2",        0xc000, 0x2000, 0x90f9c940 )
	ROM_LOAD( "yrom1",        0xe000, 0x2000, 0x55f8ab30 )

	/* no gfx1 (RAM is used) */

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "gv113fg3",     0x0000, 0x2000, 0x28071212 )	/* sprites */
	ROM_LOAD( "gv113fg2",     0x2000, 0x2000, 0x9fa3dfde )
	ROM_LOAD( "gv113fg1",     0x4000, 0x2000, 0xfb223854 )
	ROM_LOAD( "gv113fg0",     0x6000, 0x2000, 0x95762c53 )
ROM_END

ROM_START( curvebal )
	ROM_REGION( 0x10000, REGION_CPU1 )	/* 64k for code */
	ROM_LOAD( "cb-rom-3.chp", 0x8000, 0x2000, 0x72ad4d45 )
	ROM_LOAD( "cb-rom-2.chp", 0xa000, 0x2000, 0xd46c3db5 )
	ROM_LOAD( "cb-rom-1.chp", 0xc000, 0x2000, 0xeb1e08bd )
	ROM_LOAD( "cb-rom-0.chp", 0xe000, 0x2000, 0x401fc7e3 )

	ROM_REGION( 0x10000, REGION_CPU2 )	/* 64k for sound cpu */
	ROM_LOAD( "yrom.sbd",     0xe000, 0x1000, 0x4c313d9b )
	ROM_LOAD( "drom.sbd",     0xf000, 0x1000, 0xcecece88 )

	ROM_REGION( 0x2000, REGION_GFX1 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "cb-bg-0.chp",  0x0000, 0x1000, 0xd666a179 )
	ROM_LOAD( "cb-bg-1.chp",  0x1000, 0x1000, 0x5e34ff4e )

	ROM_REGION( 0x8000, REGION_GFX2 | REGIONFLAG_DISPOSE )
	ROM_LOAD( "cb-fg-3.chp",  0x0000, 0x2000, 0x9c9452fe )
	ROM_LOAD( "cb-fg-2.chp",  0x2000, 0x2000, 0x065131af )
	ROM_LOAD( "cb-fg-1.chp",  0x4000, 0x2000, 0x1b7b7f94 )
	ROM_LOAD( "cb-fg-0.chp",  0x6000, 0x2000, 0xe3a8230e )
ROM_END


static void init_reactor(void)
{
	samples_interface.samplenames = reactor_sample_names;
}

static void init_qbert(void)
{
	samples_interface.samplenames = qbert_sample_names;
}

static void init_gottlieb(void)
{
	gottlieb_sound_init();
}


GAME( 1982, reactor,  0,     reactor,  reactor,  reactor,  ROT0,   "Gottlieb", "Reactor" )
GAME( 1983, mplanets, 0,     gottlieb, mplanets, 0,        ROT270, "Gottlieb", "Mad Planets" )
GAME( 1982, qbert,    0,     gottlieb, qbert,    qbert,    ROT270, "Gottlieb", "Q*bert (US)" )
GAME( 1982, qbertjp,  qbert, gottlieb, qbert,    qbert,    ROT270, "Gottlieb (Konami license)", "Q*bert (Japan)" )
GAME( 1983, sqbert,   0,     gottlieb, qbert,    qbert,    ROT270, "Mylstar", "Faster, Harder, More Challenging Q*bert" )
GAME( 1983, qbertqub, 0,     qbertqub, qbertqub, qbert,    ROT270, "Mylstar", "Q*bert's Qubes" )
GAME( 1983, krull,    0,     krull,    krull,    0,        ROT270, "Gottlieb", "Krull" )
GAMEX(1983, mach3,    0,     mach3,    mach3,    gottlieb, ROT0,   "Mylstar", "M.A.C.H. 3", GAME_NOT_WORKING )
GAMEX(????, usvsthem, 0,     usvsthem, usvsthem, gottlieb, ROT0,   "Mylstar", "Us vs. Them", GAME_NOT_WORKING )
GAMEX(1984, 3stooges, 0,     stooges,  3stooges, gottlieb, ROT0,   "Mylstar", "Three Stooges", GAME_IMPERFECT_SOUND )
GAME( 1984, curvebal, 0,     gottlieb, curvebal, 0,        ROT270, "Mylstar", "Curve Ball" )
