/***************************************************************************

    M.A.M.E.CE3  -  Multiple Arcade Machine Emulator for Pocket PC
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAMECE3, and may only be used, modified and
    distributed under the terms of the MAME license, in "MAME.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  file.c
  
    File handling code.
    
***************************************************************************/

#include <windows.h>

#include "mamece3.h"
#include "file.h"


#define MAX_FILES 500

/***************************************************************************
 Internal structures
***************************************************************************/

#define MAXPATHS 4 /* at most 2 path entries */

typedef struct
{
    char* psPaths[MAXPATHS];
    int   iNumPaths;
   // char* psBuf;
} tDirPaths;


/***************************************************************************
 function prototypes
***************************************************************************/

void GetRomDirs(void);
void GetSampleDirs(void);
int checksum_file(FILE *f, unsigned char **p, unsigned int *size, unsigned int *crc);

/***************************************************************************
 Internal variables
***************************************************************************/

char szAltPath[FILENAME_MAX];
char szPath[MAXPATHS*2][FILENAME_MAX];
static tDirPaths RomDirPath;
static tDirPaths SamplesDirPath;

static mame_file   *file_list[MAX_FILES];

BOOL wrote_hi     = FALSE;
BOOL wrote_cfg    = FALSE;
char hifname[FILENAME_MAX]  = "";
char cfgfname[FILENAME_MAX] = "";

/***************************************************************************
 External OSD functions  
***************************************************************************/

/*
   check if roms/samples for a game exist at all
   return index+1 of the path vector component on success, otherwise 0
*/
int File_faccess(const char *newfilename, int filetype)
{
    static const char* filename;
    static int         index;

    char        name[FILENAME_MAX];
    const char* dirname;
    tDirPaths pDirPaths;
	DWORD attr;

	/* if newfilename == NULL, continue the search */
    if (newfilename == NULL)
    {
        index++;
    }
    else
    {
        index = 0;
        filename = newfilename;
    }

    switch (filetype)
    {
		case OSD_FILETYPE_ROM:
		case OSD_FILETYPE_SAMPLE:
			if (filetype == OSD_FILETYPE_ROM)
				pDirPaths = RomDirPath;
			if (filetype == OSD_FILETYPE_SAMPLE)
				pDirPaths = SamplesDirPath;

			for (; index < pDirPaths.iNumPaths; index++)
			{
				dirname = pDirPaths.psPaths[index];

				/*
					1) dirname/filename.zip - zip file / ZipMagic
					2) dirname/filename
					3) dirname/filename.zif - ZipFolders
				*/

				sprintf(name, "%s\\%s.zip", dirname, filename);
				attr = GetFileAttributes(WC(name));
				if (attr != 0xFFFFFFFF && !(attr & FILE_ATTRIBUTE_DIRECTORY))
					return index + 1;

				sprintf(name, "%s\\%s", dirname, filename);
 				attr = GetFileAttributes(WC(name));
				if (attr != (DWORD)-1 && attr == FILE_ATTRIBUTE_DIRECTORY)
					return index + 1;
			}
			break;
		case OSD_FILETYPE_HIGHSCORE:
		case OSD_FILETYPE_CONFIG:
		case OSD_FILETYPE_INPUTLOG:
			break;
		case OSD_FILETYPE_SCREENSHOT:
			return 0;
		break;
    }

    /* no match */
    return 0;
}


/* gamename holds the driver name, filename is only used for ROMs and samples. */
/* if 'write' is not 0, the file is opened for write. Otherwise it is opened */
/* for read. */
void *File_fopen(const char *gamename,const char *filename,int filetype,int write)
{
    char        name[FILENAME_MAX];
    char        subdir[25];
    char        typestr[10];
    int         i;
    int         found = 0;
    int         use_flyers = FALSE;
    mame_file   *mf = NULL;
    const char* dirname;
	tDirPaths  pDirPaths;

	/* Writing to samples and roms are not allowed */
    if (write && 
        (OSD_FILETYPE_ROM == filetype || OSD_FILETYPE_SAMPLE == filetype))
        return NULL;

    for (i = 0; i < MAX_FILES; i++)
    {
        if (file_list[i] == NULL)
        {
            file_list[i] = (mame_file *)malloc(sizeof(mame_file));
            mf = file_list[i];
            if (mf != NULL)
            {
                memset(mf, '\0', sizeof(mame_file));
                mf->index = i;
            }
            break;
        }
    }
    if (mf == NULL)
        return NULL;

    mf->access_type = ACCESS_FILE;

    switch (filetype)
    {
		case OSD_FILETYPE_ROM:
		case OSD_FILETYPE_SAMPLE:
        
			/* only for reading */
			if (write)
				break;

			if (filetype == OSD_FILETYPE_ROM)
				pDirPaths = RomDirPath;
			if (filetype == OSD_FILETYPE_SAMPLE)
				pDirPaths = SamplesDirPath;

			for (i = 0; i < pDirPaths.iNumPaths && !found; i++)
			{
				dirname = pDirPaths.psPaths[i];

				/* zip file. */
				if (!found)
				{
					sprintf(name, "%s\\%s.zip", dirname, gamename);
					if (load_zipped_file(name, filename, &mf->file_data, &mf->file_length) == 0)
					{
						mf->access_type = ACCESS_ZIP;
						mf->crc = 0; //crc32(0L, mf->file_data, mf->file_length);
						found = 1;
					}
				}

				/* rom directory */
				if (!found)
				{
					sprintf(name, "%s\\%s\\%s", dirname, gamename, filename);
					mf->fptr = fopen(name, "rb");
					if (mf->fptr)
					{
						if (filetype == OSD_FILETYPE_ROM)
						{
							if (checksum_file(mf->fptr, &mf->file_data, &mf->file_length, &mf->crc) == 0)
							{
								mf->access_type = ACCESS_RAMFILE;
								found = 1;
							}
							fclose(mf->fptr);
							mf->fptr = 0;
						}
						else
						{
							mf->access_type = ACCESS_FILE;
							found = mf->fptr != 0;
						}
					}
				}
				if (mf->fptr != NULL)
					break;
			}
			break;
		case OSD_FILETYPE_FLYER:
			use_flyers = FALSE;
		case OSD_FILETYPE_SCREENSHOT:
			return 0;
			break;

		case OSD_FILETYPE_HIGHSCORE:
		case OSD_FILETYPE_CONFIG:
			if (filetype == OSD_FILETYPE_CONFIG)
			{
				strcpy(typestr, "cfg");
				sprintf(szAltPath, "%scfg", szRootPath);
				dirname = szAltPath;
			}
			else
			if (filetype == OSD_FILETYPE_HIGHSCORE)
			{
				//techmaster if (!mame_highscore_enabled())
				//    break;
				strcpy(typestr, "hi");
           		sprintf(szAltPath, "%shi", szRootPath);
	            dirname = szAltPath;
			}

			if (!found)
			{
				CreateDirectory(WC(dirname), NULL);
				/* Try Normal file */
				sprintf(name, "%s\\%s.%s", dirname, gamename, typestr);
				if ((mf->fptr = fopen(name, write ? "wb" : "rb")) != NULL)
				{
					if (write == 1)
					{
						if (OSD_FILETYPE_HIGHSCORE == filetype)
						{
							strcpy(hifname,name);
							wrote_hi = TRUE;
						}
						else
						if (OSD_FILETYPE_CONFIG == filetype)
						{
							strcpy(cfgfname,name);
							wrote_cfg = TRUE;
						}
						/* NOTE: Don't write snapshots to bmp.zip. */
					}
					mf->access_type = ACCESS_FILE;
					found = 1;
				}
				else
				if (write == 0)
				{
					/* then zip file. */
					mf->access_type = ACCESS_ZIP;
					sprintf(name, "%s.%s", gamename, typestr);
					sprintf(subdir, "%s.zip", typestr);
					if (load_zipped_file(subdir, name, &mf->file_data, &mf->file_length) == 0)
						found = 1;
				}
			}
			break;

		case OSD_FILETYPE_INPUTLOG:
			sprintf(name, "%s", gamename);
			mf->access_type = ACCESS_FILE;
			mf->fptr = fopen(name, write ? "wb" : "rb");
			found = mf->fptr != 0;
			break;

		case OSD_FILETYPE_ARTWORK:
			/* only for reading */
			mf->fptr = 0;
			found = 0;
			break;
	}

    if (!found)
    {
        file_list[mf->index] = NULL;
        free(mf);
        mf = NULL;
    }

	return mf;
}


int File_fread(void *file, void *buffer, int length)
{
    mame_file *mf = file;
        
    switch (mf->access_type)
    {
    case ACCESS_FILE:
        return fread(buffer, 1, length, mf->fptr);

    case ACCESS_ZIP:
    case ACCESS_RAMFILE:
        if (mf->file_data)
        {
            if (length + mf->file_offset > mf->file_length)
                length = mf->file_length - mf->file_offset;
            memcpy(buffer, mf->file_offset + mf->file_data, length);
            mf->file_offset += length;
            return length;
        }
        break;
    }

    return 0;
}


int File_fwrite(void *file, const void *buffer, int length)
{
    mame_file *mf = file;
        
    switch (mf->access_type)
    {
    case ACCESS_FILE:
        return fwrite(buffer, 1, length, mf->fptr);

    case ACCESS_ZIP:
        printf("writing to zip files not supported\n");
        return 0;
    }

    printf("invalid file type %i\n", mf->access_type);
    return 0;
}



int File_fread_scatter(void *file,void *buffer,int length,int increment)
{
	return 0;
}

int File_fseek(void *file, int offset, int whence)
{
    mame_file *mf = file;
        
    switch (mf->access_type)
    {
    case ACCESS_FILE:
        return fseek(mf->fptr, offset, whence);

    case ACCESS_ZIP:
    case ACCESS_RAMFILE:
        /* seeking within the RAM image of a file */
        switch (whence)
        {
        case SEEK_SET:
            mf->file_offset = offset;
            break;

        case SEEK_CUR:
            mf->file_offset += offset;
            break;

        case SEEK_END:
            mf->file_offset = mf->file_length + offset;
            break;

        default: 
            return 0;
        }

        if (mf->file_offset < 0)
            mf->file_offset = 0;
        if (mf->file_offset > mf->file_length)
            mf->file_offset = mf->file_length;
        return 0;
    }

    printf("invalid file type %i\n", mf->access_type);
    return 0;
}


void File_fclose(void *file)
{
    mame_file *mf = file;

    switch (mf->access_type)
    {
    case ACCESS_FILE:
        fclose(mf->fptr);
//		if (wrote_hi == TRUE)   /* If hi.zip exists... */
//        {
//            /* Move the new highscore file into it */
//            ZipFile("hi.zip", hifname);
//            wrote_hi = FALSE;
//            hifname[0] = '\0';
//        }
//        if (wrote_cfg == TRUE)  /* If cfg.zip exists...*/
//        {
//            /* Move the new config file into it */
//            ZipFile("cfg.zip", cfgfname);
//            wrote_cfg = FALSE;
//            cfgfname[0] = '\0';
//        }
        break;

    case ACCESS_ZIP:
    case ACCESS_RAMFILE:
        /* freeing the file's memory allocated by the unzip code  */
        if (mf->file_data)
            free(mf->file_data);
        break;

    default:
        printf("invalid file type %i\n", mf->access_type);
    }
    
    file_list[mf->index] = NULL;
    free(mf);
}


int File_fchecksum(const char *gamename, const char *filename, unsigned int *length, unsigned int *sum)
{
    char name[FILENAME_MAX];
    int  i;
    int  found = 0;
    char *dirname;
	tDirPaths pDirPaths;
	pDirPaths = RomDirPath;
    
    for (i = 0; i < pDirPaths.iNumPaths && !found; i++)
    {
        dirname = pDirPaths.psPaths[i];
        /* rom directory */
        if (!found)
        {
			FILE *fptr;

            sprintf(name, "%s\\%s\\%s",dirname, gamename, filename);
			fptr = fopen(name, "rb");
			if (fptr)
			{
	            if (checksum_file(fptr, 0, length, sum) == 0)
	                found = 1;
				fclose(fptr);
				fptr = 0;
			}
        }
    }

    if (!found)
        return -1;
    
    return 0;
}


int File_ftell(void *file)
{
    mame_file *mf = file;
    
    switch (mf->access_type)
    {
    case ACCESS_FILE:
        return ftell(mf->fptr);

    case ACCESS_ZIP:
    case ACCESS_RAMFILE:
        return mf->file_offset;
    }

    printf("invalid file type %i\n", mf->access_type);
    return -1;
}

int File_fsize(void *file)
{
    mame_file *mf = file;
    
    switch (mf->access_type)
    {
    case ACCESS_ZIP:
    case ACCESS_RAMFILE:
        return mf->file_length;
    }

    return 0;
}

unsigned int File_fcrc(void *file)
{
    mame_file *mf = file;
    
    return mf->crc;
}


/***************************************************************************
 External functions
***************************************************************************/


/***************************************************************************
 Internal functions  
***************************************************************************/

//* This Function retrieves the current path to this program and strips off the mamece3.exe
//	This gives us the current directory :) - Techmaster
void set_root_path(void)
{
	WORD modpath[MAX_PATH];
	char *p;

	/* Get Module Path */
	GetModuleFileName(NULL, modpath, MAX_PATH);
	WideCharToMultiByte(CP_ACP, 0, 
		modpath, wcslen(modpath) + 1, 
		szRootPath, MAX_PATH - 1, 
		NULL, NULL);
	p = szRootPath + strlen(szRootPath);
	while (p != szRootPath && *p != '\\')
		*p-- = 0;

	GetRomDirs();
	GetSampleDirs();
}

void GetRomDirs(void)
{	
	RomDirPath.iNumPaths = MAXPATHS;
	
	sprintf(szPath[0], "%sroms", szRootPath);
	 RomDirPath.psPaths[0] = szPath[0];
	sprintf(szPath[1], "\\Storage Card%sroms", szRootPath);
	 RomDirPath.psPaths[1] = szPath[1];
	sprintf(szPath[2], "\\My Documents\\Roms", szRootPath);
	 RomDirPath.psPaths[2] = szPath[2];
	sprintf(szPath[3], "\\My Documents", szRootPath);
	 RomDirPath.psPaths[3] = szPath[3];

}
void GetSampleDirs(void)
{
	SamplesDirPath.iNumPaths = MAXPATHS;
		
	sprintf(szPath[4], "%sroms", szRootPath);
	 SamplesDirPath.psPaths[0] = szPath[4];
	sprintf(szPath[5], "\\Storage Card%sroms", szRootPath);
	 SamplesDirPath.psPaths[1] = szPath[5];
	sprintf(szPath[6], "\\My Documents\\samples", szRootPath);
	 SamplesDirPath.psPaths[2] = szPath[6]; 
	sprintf(szPath[7], "\\My Documents", szRootPath);
	 SamplesDirPath.psPaths[3] = szPath[7];
}


int checksum_file(FILE *f, unsigned char **p, unsigned int *size, unsigned int *crc)
{
    int length;
    unsigned char *data;

    /* determine length of file */
    if (fseek (f, 0L, SEEK_END) != 0)
    {
        fclose(f);
        return -1;
    }

    length = ftell(f);
    if (length == -1L)
    {
        fclose(f);
        return -1;
    }

    /* allocate space for entire file */
    data = (unsigned char*)malloc(length);
    if (!data)
    {
        fclose(f);
        return -1;
    }

    /* read entire file into memory */
    if (fseek(f, 0L, SEEK_SET) != 0)
    {
        free(data);
        fclose(f);
        return -1;
    }

    if (fread(data, sizeof (unsigned char), length, f) != (size_t)length)
    {
        free(data);
        fclose(f);
        return -1;
    }

    *size = length;
    *crc = 0;//crc32(0L, data, length);
    if (p)
        *p = data;
    else
        free(data);

    return 0;
}