/***************************************************************************

    M.A.M.E.CE3  -  Multiple Arcade Machine Emulator for Pocket PC
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAMECE3, and may only be used, modified and
    distributed under the terms of the MAME license, in "MAME.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  GDIDisplay.c

 ***************************************************************************/

//#define WIN32_LEAN_AND_MEAN
//#include <windows.h>

//#include <assert.h>
//#include <math.h>

//#include "mamece.h"
//#include "resource.h"


//#include "M32Util.h"
//#include "status.h"
#include <windowsx.h>

#include "Playgame.h"
#include "GDIDisplay.h"
#include "driver.h"
#include "dirty.h"
#include "uclock.h"

#ifdef GAPI
	#include "cgapi.h"
#else
#ifdef CGAMEX
	#include "cgamex.h"
#endif
#endif

/***************************************************************************
    function prototypes
 ***************************************************************************/

int				GDI_init(options_type *options);
void			GDI_exit(void);
BOOL			GDI_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);

void			DrawToScreen(void);
static void			SetPen(int pen, unsigned char red, unsigned char green, unsigned char blue);
static void			SetPaletteColors(void);
static void			AdjustPalette(void);
static int			GetBlackPen(void);
void			AdjustColorMap(void);
void			Display_MapColor(unsigned char* pRed, unsigned char* pGreen, unsigned char* pBlue);
static void			Throttle(void);

static void			update_display(void);

void			Display_WriteBitmap(struct osd_bitmap* tBitmap, PALETTEENTRY* pPalEntries);


/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDDisplay   GDIDisplay = 
{
    { GDI_init },               /* init              */
    { GDI_exit },               /* exit              */
	{ GDI_OnMessage },		/* OnMessage		 */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

#define MAKECOL(r, g, b) (((r) & 0x001F) << 10 | ((g) & 0x001F) << 5 | ((b) & 0x001F))

#define PF_15BIT			0		/* 16 x555 Format */
#define PF_16BIT			1		/* 16 565 Format */
#define PF_24BITRGB			2		/* 24 RGB Format */
#define PF_32BITRGB			3		/* 32 XRGB Format */

#define OSD_NUMPENS			(256)
#define FRAMESKIP_LEVELS	12
#define MAX_FRAMESKIP		8
#define FRAMES_TO_SKIP		20	/* skip the first few frames from the FPS calculation */
								/* to avoid counting the copyright and info screens */


//struct GDIBitmap
//{
//	int linesize;
//	int bufsize;
//	HBITMAP bitmap;
//	BITMAPINFO bi;
	// Don't add anything after BITMAPINFO.. as the RGBQUAD colors must follow directly after it!!
//};

struct tDisplay_private
{
    struct osd_bitmap*  m_pMAMEBitmap;
    struct osd_bitmap*  m_pUpdateBitmap;

	HBITMAP				m_hBitmap;
	HBITMAP				m_hOldBitmap;
	HDC					m_hMemDC;

	BYTE**				m_pScreenLines;

    RECT                m_ClientRect;
    RECT                m_DrawRect;

	BOOL				m_bStretch;
	BOOL                m_bScanlines;
    BOOL                m_bDouble;
	BOOL                m_bUseDirty;
	int					m_nScreenWidth;
	int					m_nScreenHeight;
	int					m_nScreenDepth;
	int                 m_nDepth;
	int					m_PixelFormat;


    BOOL                m_bUpdatePalette;
    HPALETTE            m_hPalette;
    PALETTEENTRY        m_PalEntries[OSD_NUMPENS];
    PALETTEENTRY        m_AdjustedPalette[OSD_NUMPENS];
	WORD				m_Pal16[OSD_NUMPENS];

    UINT                m_nUIPen;


	//from display
   /* Gamma/Brightness */
    unsigned char   m_pColorMap[OSD_NUMPENS];
    double          m_dGamma;
    int             m_nBrightness;

	/* Frames per second (FPS) info */
	BOOL            m_bShowFPS;
	int             m_nShowFPSTemp;

    /* Speed throttling / frame skipping */
	BOOL            m_bThrottled;
	BOOL            m_bAutoFrameskip;
	int             m_nFrameSkip;
	int             m_nFrameskipCounter;
    int             m_nFrameskipAdjust;
    int             m_nSpeed;
	uclock_t        m_PrevFrames[FRAMESKIP_LEVELS];
	uclock_t        m_Prev;

    /* vector game stats */
	int             m_nVecUPS;
	int             m_nVecFrameCount;

    /* average frame rate data */
	int             frames_displayed;
	uclock_t        start_time;
	uclock_t        end_time;  /* to calculate fps average on exit */
 
    /* AVI Capture params */
//    BOOL                m_bAviCapture;      /* Capture AVI mode */
//    BOOL                m_bAviRun;          /* Capturing frames */
//    int                 m_nAviShowMessage;  /* Show status if !0 */
	
	/* Device-Manufacturer ID */
	int m_nDevID;
};


/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tDisplay_private      This;

static int	m_bpp;
// moved these 4 variable to global in order to improve draw_to_screen - 11-30-01  Techmaster
static int  m_pitchX;
static int  m_pitchY;
static int	iheight;
static int	iwidth;

static int	iAdjustBase = 0;

/***************************************************************************
    External OSD functions  
 ***************************************************************************/

//
//  put here anything you need to do when the program is started. Return 0 if 
//  initialization was successful, nonzero otherwise.
//
int GDI_init(options_type *options)
{
	int i;
	
	This.m_pMAMEBitmap      = NULL;
	This.m_pUpdateBitmap    = NULL;

	This.m_pScreenLines		= NULL;
	
	This.m_bStretch         = FALSE; // options->stretch
    This.m_bScanlines       = FALSE; // options->hscan_lines; // Scanlines don't work.
    This.m_bDouble          =  2;// (options->scale == 2);

    This.m_bUseDirty        = options->use_dirty;
	This.m_nScreenDepth		= 8;//16;options->screendepth;	 // Added to correct colors - Techmaster
	This.m_nDepth           = options->depth;//8;//16;		 // Added to correct colors - Techmaster
	This.m_nUIPen           = 0;

	This.m_dGamma            = max(0, options->gamma);
    This.m_nBrightness       = max(0, options->brightness);

	This.m_bShowFPS          = options->showfps;
	This.m_nShowFPSTemp      = 0;
 
	This.m_bThrottled        = options->auto_frame_skip; //TRUE; //Techmaster
    This.m_bAutoFrameskip    = options->auto_frame_skip;
	This.m_nFrameSkip        = options->frame_skip;
	This.m_nFrameskipCounter = 0;
	This.m_nFrameskipAdjust  = 0;
	This.m_nSpeed            = 0;
	This.m_Prev              = uclock();
    
	This.m_nVecUPS           = 0;
    This.m_nVecFrameCount    = 0;
 
	This.frames_displayed    = 0;
    This.start_time          = uclock();
    This.end_time            = uclock();
	This.m_nDevID			 = options->DevID;

    for (i = 0; i < FRAMESKIP_LEVELS; i++)
        This.m_PrevFrames[i] = This.m_Prev;

    if (This.m_bAutoFrameskip == TRUE)
        This.m_nFrameSkip = 0;


	// Initialize map. //
    AdjustColorMap();

    This.m_bUpdatePalette   = FALSE;
    This.m_hPalette         = NULL;
    memset(This.m_PalEntries,      0, OSD_NUMPENS * sizeof(PALETTEENTRY));
    memset(This.m_AdjustedPalette, 0, OSD_NUMPENS * sizeof(PALETTEENTRY));
	memset(This.m_Pal16,		   0, OSD_NUMPENS * sizeof(WORD));
     	
    
    // Check for inconsistent parameters. //
    if (This.m_bScanlines == TRUE
    &&  This.m_bDouble    == FALSE)
    {
        This.m_bScanlines = FALSE;
    }

    if (This.m_nDepth < 8)
        This.m_nDepth = 8;
    else
    if (8 < This.m_nDepth)
        This.m_nDepth = 16;

    return 0;
}


//
//  put here cleanup routines to be executed when the program is terminated.
//
void GDI_exit(void)
{
//	if (This.frames_displayed > FRAMES_TO_SKIP)
//		printf("Average FPS: %f\n",
//			(double)UCLOCKS_PER_SEC / (This.end_time - This.start_time) *
//			(This.frames_displayed - FRAMES_TO_SKIP));
}



//
//  Create a bitmap. Also, call osd_clearbitmap() to appropriately initialize it to 
//  the background color.
//
struct osd_bitmap* GD_new_bitmap(int width, int height,int depth)
{
       struct osd_bitmap*  pBitmap;

    if (Machine->orientation & ORIENTATION_SWAP_XY)
    {
        int     temp;

        temp   = width;
        width  = height;
        height = temp;
    }

    pBitmap = (struct osd_bitmap*)malloc(sizeof(struct osd_bitmap));

    if (pBitmap != NULL)
    {
        unsigned char*  bm;
        int     i;
        int     safety;
        int     rdwidth;

		pBitmap->width  = width;
        pBitmap->height = height;
        pBitmap->depth  = depth;

        if (pBitmap->width > 64)
			safety = 8;
        else
           safety = 0;

        rdwidth = (width + 7) & ~7;     /* round width to a quadword */

        bm = (unsigned char*)malloc((rdwidth + 2 * safety) * (height + 2 * safety) * depth / 8);
        if (bm == NULL)
        {
            free(pBitmap);
            return NULL;
        }

        if ((pBitmap->line = malloc(height * sizeof(unsigned char *))) == 0)
        {
            free(bm);
            free(pBitmap);
            return 0;
        }

        for (i = 0; i < height; i++)
            pBitmap->line[i] = &bm[(i + safety) * (rdwidth + 2 * safety) * depth / 8 + safety];

        pBitmap->_private = bm;

        GD_clearbitmap(pBitmap);
    }

    return pBitmap;
}


//
//   Free memory allocated in create_bitmap.
//
void GD_free_bitmap(struct osd_bitmap* pBitmap)
{
	if (pBitmap != NULL)
    {
        free(pBitmap->line);
        free(pBitmap->_private);
        free(pBitmap);
    }
}


//
//  Create a display screen, or window, large enough to accomodate a bitmap
//  of the given dimensions.
//  Return a osd_bitmap pointer or 0 in case of error.
//
struct osd_bitmap* GD_create_display(int width, int height, int attributes)
{
    unsigned int    i;
    RECT            Rect;
    HMENU           hMenu;
    LOGPALETTE      LogPalette;
	DWORD			pal[255];		// Extra space for LOGPALETTE.. must follow LogPalette
    char			TitleBuf[256];
	HDC				hDC;
	HBITMAP			bm;
	int				screen_modulo;

    // Find out if display depth is 15 bpp or greater. //
	hDC = GetDC(NULL);
    This.m_nScreenDepth = GetDeviceCaps(hDC, BITSPIXEL) * GetDeviceCaps(hDC, PLANES);

#ifdef GAPI
	This.m_PixelFormat = PF_15BIT;
	This.m_hMemDC = NULL;
	m_bpp = GAPI_GetFBBpp();
#else
	#ifdef CGAMEX
	This.m_PixelFormat = PF_15BIT;
	This.m_hMemDC = NULL;
	m_bpp = GAMEX_GetFBBpp();
	#else
		This.m_PixelFormat = PF_16BIT;
		This.m_hMemDC = CreateCompatibleDC(hDC);
#endif
#endif

	// Set up direct access array //
	This.m_nScreenWidth  = GetSystemMetrics(SM_CXSCREEN);
	This.m_nScreenHeight = (GetSystemMetrics(SM_CYSCREEN));

    if (This.m_nDepth == 16 && 15 <= This.m_nScreenDepth && (attributes & VIDEO_SUPPORTS_16BIT))
        This.m_nDepth = 16;
    else
        This.m_nDepth = 8;

    if (This.m_bUseDirty == TRUE)
    {
        if (Machine->orientation & ORIENTATION_SWAP_XY)
            InitDirty(height, width, USE_DIRTYRECT);
        else
            InitDirty(width, height, USE_DIRTYRECT);
    }

    // osd_create_bitmap will swap width and height if neccesary. //
    This.m_pMAMEBitmap = GD_new_bitmap(width, height, This.m_nDepth);
    This.m_pUpdateBitmap = GD_new_bitmap(width, height, This.m_nDepth);

#ifdef GAPI
	This.m_hBitmap = This.m_hOldBitmap = NULL;
#else
	#ifdef CGAMEX
		This.m_hBitmap = This.m_hOldBitmap = NULL;
	#else
	// Create BitmapInfo //
	bi.bmiHeader.biSize          = sizeof(BITMAPINFOHEADER); 
	bi.bmiHeader.biWidth         = This.m_nScreenWidth;
	bi.bmiHeader.biHeight        = -This.m_nScreenHeight; /* Negative means "top down" */
	bi.bmiHeader.biPlanes        = 1;
	bi.bmiHeader.biBitCount      = This.m_nScreenDepth;
	bi.bmiHeader.biCompression   = 
		(This.m_nScreenDepth == 16 || This.m_nScreenDepth == 32) ? BI_BITFIELDS : BI_RGB;
	bi.bmiHeader.biSizeImage     = 0;
	bi.bmiHeader.biXPelsPerMeter = 0;
	bi.bmiHeader.biYPelsPerMeter = 0;
	bi.bmiHeader.biClrUsed       = 0;
	bi.bmiHeader.biClrImportant  = 0;

	if (This.m_nScreenDepth == 8)
	{
		for (i = 0; i < OSD_NUMPENS; i++)
			((WORD*)(bi.bmiColors))[i] = i;
	}
	if (This.m_nScreenDepth == 16)
	{
		if (This.m_PixelFormat == PF_15BIT)
		{
			((DWORD*)(bi.bmiColors))[0] = (WORD)(0x1F << 10);
			((DWORD*)(bi.bmiColors))[1] = (WORD)(0x1F << 5);
			((DWORD*)(bi.bmiColors))[2] = (WORD)(0x1F << 0);
		}
		else
		{
			((DWORD*)(bi.bmiColors))[0] = (WORD)(0x1F << 11);
			((DWORD*)(bi.bmiColors))[1] = (WORD)(0x3F << 5);
			((DWORD*)(bi.bmiColors))[2] = (WORD)(0x1F << 0);
		}
	}
	else if (This.m_nScreenDepth == 24) // Not actually used in CE..
	{
		((DWORD*)(bi.bmiColors))[0] = 0xFF << 16;
		((DWORD*)(bi.bmiColors))[1] = 0xFF << 8;
		((DWORD*)(bi.bmiColors))[2] = 0xFF << 0;
	}
	else if (This.m_nScreenDepth == 32) // Not actually used in CE..
	{
		((DWORD*)(bi.bmiColors))[0] = 0xFF << 16;
		((DWORD*)(bi.bmiColors))[1] = 0xFF << 8;
		((DWORD*)(bi.bmiColors))[2] = 0xFF << 0;
	}

	This.m_hBitmap = CreateDIBSection(hDC, &bi, 
		(This.m_nScreenDepth == 8) ? DIB_PAL_COLORS : DIB_RGB_COLORS, &bits, NULL, 0);
	ReleaseDC(NULL, hDC);  

	This.m_hOldBitmap = SelectObject(This.m_hMemDC, (HGDIOBJ)This.m_hBitmap);
#endif
#endif

	This.m_pScreenLines = (BYTE**)malloc(This.m_nScreenHeight * sizeof(BYTE*));


#ifdef CGAMEX
	screen_modulo = 512; // GAMEX_GetFBModulo();  The function call seems to get messed up??
	for (i = 0; i < This.m_nScreenHeight; i++)
		This.m_pScreenLines[i] = (BYTE*)GAMEX_GetFBAddress() + i * screen_modulo;
//#else
//	screen_modulo = ((This.m_nScreenWidth * This.m_nScreenDepth / 8) + 3) & 0xFFFFFFFC;
//	for (i = 0; i < This.m_nScreenHeight; i++)
//		This.m_pScreenLines[i] = (BYTE*)bits + i * screen_modulo;
#endif

    // Palette //
	if (This.m_nScreenDepth <= 8)
	{
	    LogPalette.palVersion    = 0x0300;
		LogPalette.palNumEntries = OSD_NUMPENS;
		This.m_hPalette = CreatePalette(&LogPalette);
	}
	else
		This.m_hPalette = NULL;


    sprintf(TitleBuf, "%s - %s", Machine->gamedrv->description, MAMECE3App.m_Name);
    SetWindowText(MAMECE3App.m_hWnd, WC(TitleBuf));

#ifdef GAPI
	GAPI_GetScreenRect(&This.m_ClientRect);
	// Added If statement to handle Ipaq3800 screen workaround 11-22-01 - Techmaster //
	if(This.m_nDevID == IPAQ_H3800)
	{
		m_pitchX = -640 >> 1;
		m_pitchY = 2 >> 1;
	}
	else
	{
		m_pitchY = (GAPI_GetFByPitch() ) >> 1;
		m_pitchX = (GAPI_GetFBxPitch() ) >> 1;
	}
#else
	#ifdef CGAMEX
		GetClientRect(MAMECE3App.m_hWnd, &This.m_ClientRect);
	#endif
#endif

	This.m_DrawRect = This.m_ClientRect;

    // Fit to screen with correct aspect ratio //
   	if (!This.m_bStretch)
	{
		This.m_DrawRect.right = This.m_pMAMEBitmap->width;
	 	This.m_DrawRect.bottom = This.m_pMAMEBitmap->height;

		if (This.m_pMAMEBitmap->width > This.m_ClientRect.right)
		{
			This.m_DrawRect.right = This.m_ClientRect.right;
			This.m_DrawRect.bottom = This.m_ClientRect.bottom *
				This.m_ClientRect.right / This.m_pMAMEBitmap->width;
		}
		
		if (This.m_pMAMEBitmap->height > This.m_ClientRect.bottom)
		{
			This.m_DrawRect.bottom = This.m_ClientRect.bottom;
			This.m_DrawRect.right = This.m_ClientRect.right *
				This.m_ClientRect.bottom / This.m_pMAMEBitmap->height;
		}

		if (This.m_DrawRect.right < This.m_ClientRect.right)
		{
			This.m_DrawRect.left = (This.m_ClientRect.right - This.m_DrawRect.right) / 2;
			This.m_DrawRect.right += This.m_DrawRect.left;
		}

		if (This.m_DrawRect.bottom < This.m_ClientRect.bottom)
		{
			This.m_DrawRect.top = (This.m_ClientRect.bottom - This.m_DrawRect.bottom) / 2;
			This.m_DrawRect.bottom += This.m_DrawRect.top;
		}
	}

    ShowWindow(MAMECE3App.m_hWnd, SW_SHOW);
    SetForegroundWindow(MAMECE3App.m_hWnd);
    InvalidateRect(MAMECE3App.m_hWnd, NULL, TRUE);
	UpdateWindow(MAMECE3App.m_hWnd);

#ifdef GAPI
	// Generate AdjustBase offset for Screen Centering of Games
	if (!(Machine->drv->video_attributes & VIDEO_TYPE_VECTOR))
	{
		if (This.m_pMAMEBitmap->width > This.m_nScreenWidth)
		{
			iAdjustBase = (This.m_pMAMEBitmap->width - This.m_nScreenWidth) / 2;
		}
		else
		{
			if (This.m_pMAMEBitmap->width < This.m_nScreenWidth)
			{
				iAdjustBase = -((This.m_nScreenWidth - This.m_pMAMEBitmap->width)  / 2);
			}
		}
		set_ui_visarea(10, 0, This.m_pMAMEBitmap->width - 10, This.m_pMAMEBitmap->height - 1);
	}
	else 
	{
		set_ui_visarea(15, 0, This.m_pMAMEBitmap->width - 5, This.m_pMAMEBitmap->height - 1);
	}
#else
	set_ui_visarea(10, 0, This.m_pMAMEBitmap->width - 10, This.m_pMAMEBitmap->height - 1);
#endif

	// Moved from draw_to_screen to improve performance - 11-30-01 Techmaster
	iheight = min(This.m_pMAMEBitmap->height,This.m_nScreenHeight);
	iwidth =  min(This.m_pMAMEBitmap->width,This.m_nScreenWidth);

	return This.m_pMAMEBitmap;
}



//
//  Set the actual display screen but don't allocate the screen bitmap.
//
int GD_set_display(int width, int height, int attributes)
{
    SetForegroundWindow(MAMECE3App.m_hWnd);
    return 0; 
}

//
//   Shut down the display
//
void GD_close_display(void)
{
	ExitDirty();
    
    osd_free_bitmap(This.m_pMAMEBitmap);
	osd_free_bitmap(This.m_pUpdateBitmap);
	free(This.m_pScreenLines);

//If GAPI or GAMEX
	SelectObject(This.m_hMemDC, (HGDIOBJ)This.m_hOldBitmap);
    DeleteObject((HGDIOBJ)This.m_hMemDC);
	DeleteObject(This.m_hBitmap);
	DeleteDC(This.m_hMemDC);
    if (This.m_hPalette)
		DeletePalette(This.m_hPalette);
//

	This.m_pMAMEBitmap = This.m_pUpdateBitmap = NULL;
	This.m_hMemDC = NULL;
	This.m_hBitmap = This.m_hOldBitmap = NULL;
	This.m_hPalette = NULL;
	This.m_pScreenLines = NULL;
}


//
//  palette is an array of 'totalcolors' R,G,B triplets. The function returns
//  in *pens the pen values corresponding to the requested colors.
//  If 'totalcolors' is 32768, 'palette' is ignored and the *pens array is filled
//  with pen values corresponding to a 5-5-5 15-bit palette
int GD_allocate_colors(unsigned int totalcolors, const unsigned char *palette, unsigned short *pens, int modifiable)
{
    unsigned int i;

    if (This.m_nDepth == 8)
    {
        if (totalcolors >= OSD_NUMPENS - 1)
        {
            int bestblack;
            int bestwhite;
            int bestblackscore;
            int bestwhitescore;

            bestblack = bestwhite = 0;
            bestblackscore = 3 * 255 * 255;
            bestwhitescore = 0;
            for (i = 0; i < totalcolors; i++)
            {
                int r, g, b, score;

                r = palette[3 * i];
                g = palette[3 * i + 1];
                b = palette[3 * i + 2];
                score = r * r + g * g + b * b;

                if (score < bestblackscore)
                {
                    bestblack      = i;
                    bestblackscore = score;
                }
                if (score > bestwhitescore)
                {
                    bestwhite      = i;
                    bestwhitescore = score;
                }
            }

            for (i = 0; i < totalcolors; i++)
                pens[i] = i;

            // map black to pen 0, otherwise the screen border will not be black //
            pens[bestblack] = 0;
            pens[0] = bestblack;

            Machine->uifont->colortable[0] = pens[bestblack];
            Machine->uifont->colortable[1] = pens[bestwhite];
            Machine->uifont->colortable[2] = pens[bestwhite];
            Machine->uifont->colortable[3] = pens[bestblack];

            This.m_nUIPen = pens[bestwhite];
        }
        else
        {
            /*
                If we have free places, fill the palette starting from the end,
                so we don't touch color 0, which is better left black.
            */
            for (i = 0; i < totalcolors; i++)
                pens[i] = OSD_NUMPENS - 1 - i;

            // As long as there are free pens, set 0 as black for GetBlackPen. //
            SetPen(0, 0, 0, 0);

            // reserve color 1 for the user interface text //
            This.m_nUIPen = 1;
            SetPen(This.m_nUIPen, 0xFF, 0xFF, 0xFF);

            Machine->uifont->colortable[0] = 0;
            Machine->uifont->colortable[1] = This.m_nUIPen;
            Machine->uifont->colortable[2] = This.m_nUIPen;
            Machine->uifont->colortable[3] = 0;
        }

        for (i = 0; i < totalcolors; i++)
        {
            SetPen(pens[i],
                   palette[3 * i],
                   palette[3 * i + 1],
                   palette[3 * i + 2]);
        }

        SetPaletteColors();
    }
    else
    {
        int r, g, b;

        for (r = 0; r < 32; r++)
            for (g = 0; g < 32; g++)
                for (b = 0; b < 32; b++)
                {
                    int r1, g1, b1;

                    r1 = (int)(31.0 * GD_get_brightness() * pow(r / 31.0, 1.0 / GD_get_gamma()) / 100.0);
                    g1 = (int)(31.0 * GD_get_brightness() * pow(g / 31.0, 1.0 / GD_get_gamma()) / 100.0);
                    b1 = (int)(31.0 * GD_get_brightness() * pow(b / 31.0, 1.0 / GD_get_gamma()) / 100.0);

                    *pens++ = MAKECOL(r1, g1, b1);
                }

        Machine->uifont->colortable[0] = 0;
        Machine->uifont->colortable[1] = MAKECOL(0xFF, 0xFF, 0xFF);
        Machine->uifont->colortable[2] = MAKECOL(0xFF, 0xFF, 0xFF);
        Machine->uifont->colortable[3] = 0;
    }

	return 0;
}


//
//  Change the color of the pen.
//
void GD_modify_pen(int pen, unsigned char red, unsigned char green, unsigned char blue)
{
//   assert(This.m_nDepth == 8); // Shouldn't modify pen in 16 bit color depth game mode! //

//   assert(0 <= pen && pen < OSD_NUMPENS);

    if (This.m_PalEntries[pen].peRed    == red
    &&  This.m_PalEntries[pen].peGreen  == green
    &&  This.m_PalEntries[pen].peBlue   == blue)
        return;

    SetPen(pen, red, green, blue);

    This.m_bUpdatePalette = TRUE;
}


//
//  Get the color of a pen.
//
void GD_get_pen(int pen, unsigned char* pRed, unsigned char* pGreen, unsigned char* pBlue)
{
    if (This.m_nDepth == 8)
    {
//        assert(0 <= pen && pen < OSD_NUMPENS);

        if (OSD_NUMPENS <= pen)
            pen = 0;

        *pRed   = This.m_PalEntries[pen].peRed;
        *pGreen = This.m_PalEntries[pen].peGreen;
        *pBlue  = This.m_PalEntries[pen].peBlue;
    }
    else
    {
        *pRed   = (pen >> 7) & 0xF8;
        *pGreen = (pen >> 2) & 0xF8;
        *pBlue  = (pen << 3) & 0xF8;
    }
}


/*
    Update the display.
*/
void GD_update_display(void)
{
    int need_to_clear_bitmap = 0;

    if (GD_skip_this_frame() == 0)
    {   
        if (This.m_nShowFPSTemp)
        {
            This.m_nShowFPSTemp--;
            if ((This.m_bShowFPS == FALSE)
            &&  (This.m_nShowFPSTemp == 0))
            {
                need_to_clear_bitmap = 1;
            }
        }
        
        /* Wait until it's time to update the screen. */
        Throttle();
      
		update_display();

//        if (need_to_clear_bitmap)
//            GD_clearbitmap(Machine->scrbitmap);

		SwitchDirty();

//        if (need_to_clear_bitmap)
//            GD_clearbitmap(Machine->scrbitmap);

        if (This.m_bThrottled
        &&  This.m_bAutoFrameskip
        &&  This.m_nFrameskipCounter == 0)
        {
            if (This.m_nSpeed < 60)
            {
                This.m_nFrameskipAdjust = 0;
                This.m_nFrameSkip += 3;
                if (This.m_nFrameSkip > MAX_FRAMESKIP)
                    This.m_nFrameSkip = MAX_FRAMESKIP;
            }
            else
            if (This.m_nSpeed < 80)
            {
                This.m_nFrameskipAdjust = 0;
                This.m_nFrameSkip += 2;
                if (This.m_nFrameSkip >= MAX_FRAMESKIP)
                    This.m_nFrameSkip = MAX_FRAMESKIP;
            }
            else
            if (This.m_nSpeed < 99) /* allow 99% speed */
            {
                This.m_nFrameskipAdjust = 0;
                /* don't push frameskip too far if we are close to 100% speed */
                if (This.m_nFrameSkip < 8)
                    This.m_nFrameSkip++;
            }
            else
            if (100 <= This.m_nSpeed)
            {
                /* increase frameskip quickly, decrease it slowly */
                This.m_nFrameskipAdjust++;
                if (This.m_nFrameskipAdjust >= 3)
                {
                    This.m_nFrameskipAdjust = 0;
                    if (This.m_nFrameSkip > 0)
                        This.m_nFrameSkip--;
                }
            }
		}
    }

    /*
        Adjust frameskip.
    */
    if (input_ui_pressed(IPT_UI_FRAMESKIP_INC))
    {
        if (This.m_bAutoFrameskip == TRUE)
        {
            This.m_bAutoFrameskip = FALSE;
            This.m_nFrameSkip     = 0;
        }
        else
        {
            if (This.m_nFrameSkip == FRAMESKIP_LEVELS - 1)
            {
                This.m_nFrameSkip     = 0;
                This.m_bAutoFrameskip = TRUE;
            }
            else
                This.m_nFrameSkip++;
        }

        if (This.m_bShowFPS == FALSE)
            This.m_nShowFPSTemp = 2 * Machine->drv->frames_per_second;

        /* reset the frame counter every time the frameskip key is pressed, so */
        /* we'll measure the average FPS on a consistent status. */
        This.frames_displayed = 0;
    }

    if (input_ui_pressed(IPT_UI_FRAMESKIP_DEC))
    {
        if (This.m_bAutoFrameskip == TRUE)
        {
            This.m_bAutoFrameskip = FALSE;
            This.m_nFrameSkip     = FRAMESKIP_LEVELS - 1;
        }
        else
        {
            if (This.m_nFrameSkip == 0)
                This.m_bAutoFrameskip = TRUE;
            else
                This.m_nFrameSkip--;
        }

        if (This.m_bShowFPS == FALSE)
            This.m_nShowFPSTemp = 2 * Machine->drv->frames_per_second;

        /* reset the frame counter every time the frameskip key is pressed, so */
        /* we'll measure the average FPS on a consistent status. */
        This.frames_displayed = 0;
    }

    /*
        Toggle throttle.
    */
    if (input_ui_pressed(IPT_UI_THROTTLE))
    {
        This.m_bThrottled = (This.m_bThrottled == TRUE) ? FALSE : TRUE;

        /* reset the frame counter every time the throttle key is pressed, so */
        /* we'll measure the average FPS on a consistent status. */
        This.frames_displayed = 0;
    }

    This.m_nFrameskipCounter = (This.m_nFrameskipCounter + 1) % FRAMESKIP_LEVELS;
}


// Set the bitmap to black.
void GD_clearbitmap(struct osd_bitmap* bitmap)
{
    int i;

    for (i = 0; i < bitmap->height; i++)
        memset(bitmap->line[i], GetBlackPen(),
               bitmap->width * (bitmap->depth / 8));

    if (bitmap == Machine->scrbitmap)
    {
        extern int bitmap_dirty;
        bitmap_dirty = 1;

        osd_mark_dirty(0, 0, bitmap->width - 1, bitmap->height - 1, 1);
    }
}


void GD_mark_dirty(int x1, int y1, int x2, int y2, int ui)
{
    if (This.m_bUseDirty == TRUE)
		MarkDirty(x1, y1, x2, y2, ui);
}


int GD_skip_this_frame(void)
{
    static const int skiptable[FRAMESKIP_LEVELS][FRAMESKIP_LEVELS] =
    {
        { 0,0,0,0,0,0,0,0,0,0,0,0 },
        { 0,0,0,0,0,0,0,0,0,0,0,1 },
        { 0,0,0,0,0,1,0,0,0,0,0,1 },
        { 0,0,0,1,0,0,0,1,0,0,0,1 },
        { 0,0,1,0,0,1,0,0,1,0,0,1 },
        { 0,1,0,0,1,0,1,0,0,1,0,1 },
        { 0,1,0,1,0,1,0,1,0,1,0,1 },
        { 0,1,0,1,1,0,1,0,1,1,0,1 },
        { 0,1,1,0,1,1,0,1,1,0,1,1 },
        { 0,1,1,1,0,1,1,1,0,1,1,1 },
        { 0,1,1,1,1,1,0,1,1,1,1,1 },
        { 0,1,1,1,1,1,1,1,1,1,1,1 }
    };

	//assert(0 <= This.m_nFrameSkip && This.m_nFrameSkip < FRAMESKIP_LEVELS);
	//assert(0 <= This.m_nFrameskipCounter && This.m_nFrameskipCounter < FRAMESKIP_LEVELS);

    return skiptable[This.m_nFrameSkip][This.m_nFrameskipCounter];
}


// control keyboard leds or other indicators //
void GD_led_w(int led, int on)
{
//    StatusWrite(led, on & 1);
}

void GD_set_gamma(float gamma)
{
    //assert(OSDDisplay.set_gamma != 0);
	if (Machine->scrbitmap->depth == 8)
	{
		if (This.m_dGamma < 0.2) This.m_dGamma = 0.2;
		if (This.m_dGamma > 3.0) This.m_dGamma = 3.0;

		This.m_dGamma = gamma;
		AdjustColorMap();
	}
	AdjustPalette();
}


float GD_get_gamma()
{
    return (float)This.m_dGamma;
}


void GD_set_brightness(int brightness)
{
	//assert(OSDDisplay.set_brightness != 0);
	if (Machine->scrbitmap->depth == 8)
	{
		This.m_nBrightness = brightness;
		AdjustColorMap();
	}
	AdjustPalette();
}

int GD_get_brightness()
{
    return This.m_nBrightness;
}


void GD_save_snapshot()
{
	// Disabled because we don't use it in MameCE - Techmaster
	//if (This.m_bAviCapture)
    //{
    //    This.m_nAviShowMessage = 10;        // show message for 10 frames //
	//	This.m_bAviRun = !This.m_bAviRun;   // toggle capture on/off //
	//}
    //else
    //    Display_WriteBitmap(This.m_pMAMEBitmap, This.m_PalEntries);
}


BOOL GDI_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
	PAINTSTRUCT     ps;
	HPALETTE		hOldPalette;

	switch (Msg)
    {
		case WM_COMMAND:
			return TRUE;
		case WM_PAINT:
		{
			BOOL			ok;
			HDC				memdc;
			HBITMAP			bm, oldbm;
			DWORD			ticks;

			BeginPaint(hWnd, &ps);

			if (This.m_hMemDC)
			{

				if (This.m_nScreenDepth == 8)
				{
					hOldPalette = SelectPalette(ps.hdc, This.m_hPalette, FALSE);
					RealizePalette(ps.hdc);
					SelectPalette(ps.hdc, hOldPalette, FALSE);
				}

				PatBlt(ps.hdc, 0, 0, This.m_nScreenWidth, This.m_nScreenHeight, BLACKNESS);
				GD_clearbitmap(This.m_pUpdateBitmap);
				GD_mark_dirty(0, 0, This.m_pUpdateBitmap->width, This.m_pUpdateBitmap->height, 0);

				DrawToScreen();
			}
			EndPaint(hWnd, &ps); 
			return TRUE;
		}
		case WM_SIZE:
			return TRUE;
		case WM_DRAWITEM:
			return TRUE;
    }
    return FALSE;
}


/***************************************************************************
    Internal functions
 ***************************************************************************/

//////////////////////////////////////////
//    Draws MAMEBitmap to the display.  //
//////////////////////////////////////////
void DrawToScreen(void)
{
	register int lineloop, pixel;
	WORD *pal16 =This.m_Pal16;

#ifdef GAPI

	register unsigned char *base;
	register unsigned short * DestLine;
	register unsigned short * SourceLine;

	// added If statement to handle iPaq3800 workaround - 11-22-01 - Techmaster //
	if(This.m_nDevID == IPAQ_H3800)
	{
		SourceLine = (unsigned short *)0xac0755a0;
	}
	else
	{
		SourceLine = (unsigned short *)GAPI_BeginDraw();
	}
	
	if (SourceLine == NULL)
		return;	// NOT OK TO DRAW

	if (This.m_bUseDirty && (Machine->drv->video_attributes &VIDEO_SUPPORTS_DIRTY))
	{
		// Move the Game Display below the Main Command Title Bar - Techmaster //
		SourceLine += (m_pitchY * 31);



		for (lineloop = 0; lineloop < iheight; lineloop++)
		{
			DestLine = SourceLine;
			base = (This.m_pMAMEBitmap->line[lineloop] + iAdjustBase);
			if (IsDirtyLine(lineloop))
			{
				for (pixel = 0; pixel < iwidth; pixel++)
				{
					*DestLine = *(pal16 + *base++);
					DestLine += m_pitchX; 
				}
			}
			SourceLine += m_pitchY;
		}
	}			
	else
	{
		// Move the Game Display below the Main Command Title Bar - Techmaster //
		SourceLine += (m_pitchY * 30);

		for (lineloop = 0; lineloop < iheight; /*lineloop+=2*/lineloop++)
		{
			DestLine = (SourceLine += m_pitchY);
			base = (This.m_pMAMEBitmap->line[lineloop] + iAdjustBase);
			for (pixel = 0; pixel < iwidth; pixel++)
			{
				*DestLine = *(pal16 + *base++);
				DestLine += m_pitchX; 
			}
			//SourceLine += m_pitchY;
		}
	}
	
	if(This.m_nDevID != IPAQ_H3800)
	{
		GAPI_EndDraw();
	}

#else
	#ifdef CGAMEX

		DWORD *m8, *d16;

		GAMEX_BeginDraw();

		iwidth = min(This.m_nScreenWidth >> 1, This.m_pMAMEBitmap->width >> 1);
		for (lineloop = 0; lineloop < iheight; lineloop++)
		{
			m8  = (DWORD*)((DWORD)This.m_pMAMEBitmap->line[lineloop] & 0xFFFFFFFC);
			d16 = (DWORD*)(((DWORD)This.m_pScreenLines[lineloop + This.m_DrawRect.top] + (This.m_DrawRect.left << 1)) & 0xFFFFFFFC);
			for (pixel = 0; pixel < iwidth; pixel++)	// Blocks are each 32 pixels
			{
				d16[0] = pal16[((BYTE*)m8)[0]] | (pal16[((BYTE*)m8)[1]] << 16);
				d16[1] = pal16[((BYTE*)m8)[2]] | (pal16[((BYTE*)m8)[3]] << 16);
				m8++;
				d16 += 2;
			}
		}
		GAMEX_EndDraw();

	#endif
#endif

}


static void SetPen(int pen, unsigned char red, unsigned char green, unsigned char blue)
{
	//assert(This.m_nDepth == 8); // Shouldn't set pen in 16 bit color depth game mode! //

    //assert(0 <= pen && pen < OSD_NUMPENS);

    This.m_PalEntries[pen].peRed    = red;
    This.m_PalEntries[pen].peGreen  = green;
    This.m_PalEntries[pen].peBlue   = blue;
    This.m_PalEntries[pen].peFlags  = 0;//PC_NOCOLLAPSE; // Not needed in CE

    memcpy(&This.m_AdjustedPalette[pen], &This.m_PalEntries[pen], sizeof(PALETTEENTRY));
    Display_MapColor(&This.m_AdjustedPalette[pen].peRed,
                     &This.m_AdjustedPalette[pen].peGreen,
                     &This.m_AdjustedPalette[pen].peBlue);
}


static void SetPaletteColors(void)
{
	if (This.m_hPalette && This.m_nDepth == 8)
		SetPaletteEntries(This.m_hPalette, 0, OSD_NUMPENS, This.m_AdjustedPalette);    
	else if (This.m_nDepth == 8 && This.m_nScreenDepth == 16)
	{
		int i;

		if (This.m_PixelFormat == PF_15BIT)
		{
			for (i = 0; i < OSD_NUMPENS; i++)
			{
				This.m_Pal16[i] = 
					((This.m_AdjustedPalette[i].peRed >> 3) << 11) |  //11 was 10 - Darren - These are best settings-Techmaster
					((This.m_AdjustedPalette[i].peGreen >> 2) << 5) | //2 was 3 - Darren
					((This.m_AdjustedPalette[i].peBlue >> 3) << 0);   //no change - Darren
			}
		}
		else
		{
			for (i = 0; i < OSD_NUMPENS; i++)
			{
				This.m_Pal16[i] = 
					((This.m_AdjustedPalette[i].peRed >> 3) << 11) |
					((This.m_AdjustedPalette[i].peGreen >> 2) << 5) |
					((This.m_AdjustedPalette[i].peBlue >> 3) << 0);
			}
		}
	}
}


static void AdjustPalette(void)
{
    if (This.m_nDepth == 8)
    {
        UINT i;
        
        memcpy(This.m_AdjustedPalette, This.m_PalEntries, OSD_NUMPENS * sizeof(PALETTEENTRY));

        for (i = 0; i < OSD_NUMPENS; i++)
        {
            if (i != This.m_nUIPen) // Don't map the UI pen. //
                Display_MapColor(&This.m_AdjustedPalette[i].peRed,
                                 &This.m_AdjustedPalette[i].peGreen,
                                 &This.m_AdjustedPalette[i].peBlue);
        }

        This.m_bUpdatePalette = TRUE;
    }
}


static int GetBlackPen(void)
{
    if (This.m_nDepth == 8)
    {
        int i;
        for	(i = 0; i < OSD_NUMPENS; i++)
        {
            if (This.m_PalEntries[i].peRed   == 0
            &&  This.m_PalEntries[i].peGreen == 0
            &&  This.m_PalEntries[i].peBlue  == 0)
            {
                return i;
            }
        }
    }

    return 0;
}


void AdjustColorMap(void)
{
    int i;
    
    for (i = 0; i < OSD_NUMPENS; i++)
    {
        This.m_pColorMap[i] = (unsigned char)((OSD_NUMPENS - 1.0)
                            * (This.m_nBrightness / 100.0)
                            * pow(((double)i / (OSD_NUMPENS - 1.0)), 1.0 / This.m_dGamma));
    }
}


void Display_MapColor(unsigned char* pRed, unsigned char* pGreen, unsigned char* pBlue)
{
    *pRed   = This.m_pColorMap[(unsigned char)*pRed];
    *pGreen = This.m_pColorMap[(unsigned char)*pGreen];
    *pBlue  = This.m_pColorMap[(unsigned char)*pBlue];
}


static void Throttle(void)
{
    static const int waittable[FRAMESKIP_LEVELS][FRAMESKIP_LEVELS] =
    {
        { 1,1,1,1,1,1,1,1,1,1,1,1 },
        { 2,1,1,1,1,1,1,1,1,1,1,0 },
        { 2,1,1,1,1,0,2,1,1,1,1,0 },
        { 2,1,1,0,2,1,1,0,2,1,1,0 },
        { 2,1,0,2,1,0,2,1,0,2,1,0 },
        { 2,0,2,1,0,2,0,2,1,0,2,0 },
        { 2,0,2,0,2,0,2,0,2,0,2,0 },
        { 2,0,2,0,0,3,0,2,0,0,3,0 },
        { 3,0,0,3,0,0,3,0,0,3,0,0 },
        { 4,0,0,0,4,0,0,0,4,0,0,0 },
        { 6,0,0,0,0,0,6,0,0,0,0,0 },
        {12,0,0,0,0,0,0,0,0,0,0,0 }
    };
    uclock_t curr;
    int i;
    int already_synced = 0;

    if (This.m_bThrottled == TRUE)
    {
		uclock_t target,target2;

        curr = uclock();

		if (already_synced == 0)
        {
			/* wait only if the audio update hasn't synced us already */

            /* wait until enough time has passed since last frame... */
			target = This.m_Prev +
				waittable[This.m_nFrameSkip][This.m_nFrameskipCounter] * UCLOCKS_PER_SEC / Machine->drv->frames_per_second;

            /* ... OR since FRAMESKIP_LEVELS frames ago. This way, if a frame takes */
            /* longer than the allotted time, we can compensate in the following frames. */
            target2 = This.m_PrevFrames[This.m_nFrameskipCounter] +
				FRAMESKIP_LEVELS * UCLOCKS_PER_SEC / Machine->drv->frames_per_second;

                if (target - target2 > 0)
                    target = target2;

                if (curr - target < 0)
                {
                    do
                    {
                        curr = uclock();
                    } while (curr - target < 0);
                }
          }
	}
    else
    {
        curr = uclock();
    }

    /* for the FPS average calculation */
    if (++This.frames_displayed == FRAMES_TO_SKIP)
        This.start_time = curr;
    else
        This.end_time = curr;

    if (This.m_nFrameskipCounter == 0 && curr - This.m_PrevFrames[This.m_nFrameskipCounter])
    {
        uclock_t divdr;

        divdr = Machine->drv->frames_per_second * (curr - This.m_PrevFrames[This.m_nFrameskipCounter]) / (100 * FRAMESKIP_LEVELS);
        This.m_nSpeed = MAMECE3App.m_nSpeed = (int)((UCLOCKS_PER_SEC + divdr / 2) / divdr);
    }

    This.m_Prev = curr;
    for (i = 0; i < waittable[This.m_nFrameSkip][This.m_nFrameskipCounter]; i++)
        This.m_PrevFrames[(This.m_nFrameskipCounter + FRAMESKIP_LEVELS - i) % FRAMESKIP_LEVELS] = curr;

    This.m_nVecFrameCount += waittable[This.m_nFrameSkip][This.m_nFrameskipCounter];
    if (This.m_nVecFrameCount >= Machine->drv->frames_per_second)
    {
        extern int vector_updates; /* avgdvg_go()'s per Mame frame, should be 1 */

        This.m_nVecFrameCount = 0;
        This.m_nVecUPS = vector_updates;
        vector_updates = 0;
    }

    /*
        Show frames per second.
    */
    if (This.m_bShowFPS == TRUE || This.m_nShowFPSTemp != 0 /* || This.m_bAviCapture*/)
    {
        int fps;
        int divdr;
		char buf[32];

        divdr = 100 * FRAMESKIP_LEVELS;
        fps = (Machine->drv->frames_per_second * (FRAMESKIP_LEVELS - This.m_nFrameSkip) * This.m_nSpeed + (divdr / 2)) / divdr;
        
		// Fix Show Framerate - Techmaster
		sprintf(buf,"%2d%3d%%(%2d/%.0f)    ",This.m_nFrameSkip,This.m_nSpeed,fps,Machine->drv->frames_per_second);
		ui_text(buf,Machine->uiwidth-strlen(buf)*Machine->uifontwidth,0);
	}
}


//
//  Update the display.
//
static void update_display(void)
{
	static int ticks, fticks;

	if (This.m_bUpdatePalette == TRUE)
		SetPaletteColors();

		ticks = GetTickCount();

		DrawToScreen();

		ticks = GetTickCount() - ticks;
		fticks = GetTickCount() - fticks;

		This.m_bUpdatePalette = FALSE;

		fticks = GetTickCount();

	MAMECE3_ProcessMessages();
}





//////////Not used right now
//////////

#ifdef PNG_SAVE_SUPPORT

void Display_WriteBitmap(struct osd_bitmap* tBitmap, PALETTEENTRY* pPalEntries)
{
	//removed not used in MameCE
}
#endif
